-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

local thModName = g_currentModName
local thModPath = g_currentModDirectory
source(thModPath .. "scripts/core/THCore.lua")
source(thModPath .. "scripts/core/managers/THMapTypeManager.lua")
THPFConfig = {}
local THPFConfig_mt = THUtils.createClass(THPFConfig, THCore)
local debugFlagId = THDebugUtil.createFlagId("THPFConfig", true)
THPFConfig.debugFlagId = debugFlagId
THPFConfig.SPRAY_TYPE_GROUP = {
    FERTILIZER = 1,
    MANURE = 2,
    DIGESTATE = 3,
    LIME = 4,
    HERBICIDE = 5
}
THUtils.createEnumTable(THPFConfig.SPRAY_TYPE_GROUP)
THPFConfig.MESSAGE = {}
local function initScript()
    local self = THPFConfig.new()
    if self ~= nil then
        _G.g_thPFConfig = self
    end
end
function THPFConfig.new(name, className, modName, customMt)
    name = name or "thPFConfig"
    className = className or "THPFConfig"
    modName = modName or thModName
    customMt = customMt or THPFConfig_mt
    local self = THCore.new(name, className, modName, customMt)
    if self ~= nil then
        self.thMapTypeManager = g_thMapTypeManager
        self.fillTypeManager = g_fillTypeManager
        self.sprayTypeManager = g_sprayTypeManager
        self.precisionFarming = {
            isFound = false
        }
        self.extendedSprayTypes = {
            byId = {},
            byIndex = {},
            bySprayTypeIndex = {},
            byGroup = {},
            onLoadValues = {},
            modOnLoadValues = {},
            isModDataLoaded = false
        }
        local sprayTypeGroups = THPFConfig.SPRAY_TYPE_GROUP:getValues()
        for _, groupIndex in pairs(sprayTypeGroups) do
            self.extendedSprayTypes.byGroup[groupIndex] = {}
        end
        return self
    end
end
function THPFConfig.onPreInit(self, ...)
    THUtils.callSuperClass(self, "onPreInit", true, ...)
    local pfModData = self.thirdPartyMods.precisionFarming
    if pfModData ~= nil and pfModData.target ~= nil then
        local pfModEnv = pfModData.environment
        local PrecisionFarming = pfModEnv.PrecisionFarming
        if PrecisionFarming ~= nil then
            THUtils.setFunctionHook("Utils", "getFilename", false, false, self, THPFConfig.inj_getFilename)
            local pfData = self.precisionFarming
            pfData.target = pfModData.target
            pfData.modData = pfModData
            pfData.modEnv = pfModData.environment
            pfData.oldBasePath = PrecisionFarming.BASE_DIRECTORY
            pfData.newBasePath = pfData.oldBasePath
            pfData.isFound = true
        end
        source(self.modPath .. "scripts/maps/THPHMap.lua")
        source(self.modPath .. "scripts/maps/THNitrogenMap.lua")
        source(self.modPath .. "scripts/objects/THStorage.lua")
        source(self.modPath .. "scripts/objects/THLoadingStation.lua")
        source(self.modPath .. "scripts/objects/THUnloadingStation.lua")
        source(self.modPath .. "scripts/gui/THExtendedSprayerHUDExtension.lua")
        self:addSpecialization(THSpecType.VEHICLE, "thFillUnit", "THFillUnitSpec", self.modName, "scripts/vehicles/specializations/THFillUnitSpec.lua")
        self:addSpecialization(THSpecType.VEHICLE, "thSprayer", "THSprayerSpec", self.modName, "scripts/vehicles/specializations/THSprayerSpec.lua")
    end
end
function THPFConfig.onSetMissionInfo(self, mission, missionInfo, missionDynamicInfo, ...)
    THUtils.callSuperClass(self, "onSetMissionInfo", true, mission, missionInfo, missionDynamicInfo, ...)
    self:loadMapData()
    local modsArray, numLoadedMods = self.thModManager:getLoadedMods()
    local modDescXMLKey = "modDesc." .. self.xmlKey
    if modsArray ~= nil and numLoadedMods > 0 then
        for modIndex = 1, numLoadedMods do
            local modData = modsArray[modIndex]
            if modData.file ~= nil and modData.file ~= ""
                and modData.path ~= nil and modData.path ~= ""
            then
                if THUtils.getFileExists(modData.file) then
                    local modDescXMLFile = THUtils.loadXMLFile("ModDescXML", modData.file)
                    if modDescXMLFile ~= nil then
                        self:loadModSprayTypesFromXML(modDescXMLFile, modDescXMLKey, modData.name, modData.path)
                        self:loadSprayTypeMappingFromXML(modDescXMLFile, modDescXMLKey, modData.name, modData.path)
                        THUtils.deleteXMLFile(modDescXMLFile)
                    end
                end
            end
        end
    end
end
function THPFConfig.onLoadMapFinished(self, mission, ...)
    THUtils.callSuperClass(self, "onLoadMapFinished", true, mission, ...)
    self.thMapTypeManager:initMapTypeData()
    self:finalizeExtendedSprayTypes()
    self.DEFAULT_FILLTYPE = self.thMapTypeManager:createFillTypeConstants()
    self.DEFAULT_SPRAYTYPE = self.thMapTypeManager:createSprayTypeConstants()
    if THDebugUtil.getIsEnabled(debugFlagId) then
        THUtils.displayMsg("Precision Farming configuration:")
        THDebugUtil.printTable(self.precisionFarming)
        THUtils.displayMsg("Extended spray types:")
        THDebugUtil.printTable(self.extendedSprayTypes)
        for key, value in pairs(self.extendedSprayTypes) do
            if type(value) == THValueType.TABLE then
                THUtils.displayMsg("[%s]:", key)
                if key == "onLoadValues"
                    or key == "modOnLoadValues"
                then
                    THDebugUtil.printTable(value, 5)
                else
                    THDebugUtil.printTable(value, 1)
                end
            end
        end
    end
end
function THPFConfig.loadFromMapXML(self, xmlFile, xmlKey, customEnv, baseDirectory, ...)
    local success = THUtils.callSuperClass(self, "loadFromMapXML", true, xmlFile, xmlKey, customEnv, baseDirectory, ...)
    if success then
        local pfData = self.precisionFarming
        local mapPath = self.mapData.path
        local isModMap = self.mapData.isModMap
        if pfData.isFound then
            local PrecisionFarming = pfData.modEnv.PrecisionFarming
            if PrecisionFarming ~= nil and PrecisionFarming.BASE_DIRECTORY ~= nil
                and pfData.oldBasePath == PrecisionFarming.BASE_DIRECTORY
            then
                local newConfigPath = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, xmlKey, "#path")
                if newConfigPath ~= nil then
                    if isModMap and mapPath ~= nil and mapPath ~= "" then
                        if newConfigPath == "" then
                            pfData.newBasePath = mapPath
                        else
                            if string.sub(newConfigPath, -1) ~= "/" then
                                newConfigPath = newConfigPath .. "/"
                            end
                            pfData.newBasePath = mapPath .. newConfigPath
                        end
                        if pfData.newBasePath ~= pfData.oldBasePath then
                            PrecisionFarming.BASE_DIRECTORY = pfData.newBasePath
                            THUtils.displayMsg("Precision Farming configuration routed to: %s", PrecisionFarming.BASE_DIRECTORY)
                        end
                    end
                end
            end
        end
    end
    if success then
        success = self:loadSprayTypeMappingFromXML(xmlFile, xmlKey, customEnv, baseDirectory)
    end
    return success
end
function THPFConfig.loadModSprayTypesFromXML(self, xmlFile, xmlKey, customEnv, baseDirectory)
    if THUtils.argIsValid(type(xmlFile) == THValueType.TABLE, "xmlFile", xmlFile)
        and THUtils.argIsValid(type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
        and THUtils.argIsValid(type(customEnv) == THValueType.STRING, "customEnv", customEnv)
        and THUtils.argIsValid(type(baseDirectory) == THValueType.STRING, "baseDirectory", baseDirectory)
    then
        local pfData = self.precisionFarming
        local modOnLoadValues = self.extendedSprayTypes.modOnLoadValues
        local success = true
        local function createModOnLoadData(pSprayTypeName)
            local sprayTypeId = pSprayTypeName:upper()
            local modOnLoadData = modOnLoadValues[sprayTypeId]
            if modOnLoadData == nil then
                modOnLoadData = {
                    id = sprayTypeId,
                    applicationRate = {
                        soilRates = {}
                    },
                    hasApplicationRate = false,
                    needsRegistration = false
                }
                modOnLoadValues[sprayTypeId] = modOnLoadData
            end
            return modOnLoadData
        end
        local sprayTypesKey = xmlKey .. ".sprayTypes"
        if THUtils.hasXMLProperty(xmlFile, sprayTypesKey) then
            xmlFile:iterate(sprayTypesKey .. ".sprayType", function(_, pSprayTypeKey)
                local sprayTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, pSprayTypeKey, "#name")
                local typeOfSpray = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, pSprayTypeKey, "#type")
                local litersPerSecond = THUtils.getXMLValue(xmlFile, XMLValueType.FLOAT, pSprayTypeKey, "#litersPerSecond")
                local sprayGroundType = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, pSprayTypeKey, "#sprayGroundType")
                if sprayTypeName ~= nil and sprayTypeName ~= "" then
                    local modOnLoadData = createModOnLoadData(sprayTypeName)
                    if modOnLoadData ~= nil then
                        if typeOfSpray ~= nil then
                            modOnLoadData.type = typeOfSpray
                        end
                        if litersPerSecond ~= nil then
                            modOnLoadData.litersPerSecond = litersPerSecond
                        end
                        if sprayGroundType ~= nil then
                            modOnLoadData.groundType = sprayGroundType
                        end
                        modOnLoadData.needsRegistration = true
                    end
                end
            end)
        end
        if pfData.isFound then
            local nMapKey = xmlKey .. ".nitrogenMap"
            local applicationRatesKey = nMapKey .. ".applicationRates"
            if THUtils.hasXMLProperty(xmlFile, applicationRatesKey) then
                xmlFile:iterate(applicationRatesKey .. ".applicationRate", function(_, pApplicationRateKey)
                    local sprayTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, pApplicationRateKey, "#fillType")
                    local adjustToFruit = THUtils.getXMLValue(xmlFile, XMLValueType.BOOL, pApplicationRateKey, "#autoAdjustToFruit")
                    local regularRate = THUtils.getXMLValue(xmlFile, XMLValueType.FLOAT, pApplicationRateKey, "#regularRate")
                    if sprayTypeName ~= nil and sprayTypeName ~= "" then
                        local modOnLoadData = createModOnLoadData(sprayTypeName)
                        if modOnLoadData ~= nil then
                            local applicationRateData = modOnLoadData.applicationRate
                            local areSoilRatesCleared = false
                            if adjustToFruit ~= nil then
                                applicationRateData.adjustToFruit = adjustToFruit
                            end
                            if regularRate ~= nil then
                                applicationRateData.regularRate = regularRate
                            end
                            modOnLoadData.hasApplicationRate = true
                            xmlFile:iterate(pApplicationRateKey .. ".soil", function(_, pSoilRateKey)
                                local soilTypeIndex = THUtils.getXMLValue(xmlFile, XMLValueType.INT, pSoilRateKey, "#soilTypeIndex")
                                local soilRate = THUtils.getXMLValue(xmlFile, XMLValueType.FLOAT, pSoilRateKey, "#rate")
                                if soilTypeIndex ~= nil and soilTypeIndex >= 0 then
                                    if not areSoilRatesCleared then
                                        THUtils.clearTable(applicationRateData.soilRates)
                                        areSoilRatesCleared = true
                                    end
                                    local soilRateData = {
                                        soilTypeIndex = soilTypeIndex,
                                        soilRate = soilRate
                                    }
                                    table.insert(applicationRateData.soilRates, soilRateData)
                                end
                            end)
                        end
                    end
                end)
            end
            local fertilizerUsageKey = nMapKey .. ".fertilizerUsage"
            if THUtils.hasXMLProperty(xmlFile, fertilizerUsageKey) then
                xmlFile:iterate(fertilizerUsageKey .. ".nAmount", function(_, pNAmountKey)
                    local sprayTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, pNAmountKey, "#fillType")
                    local nAmount = THUtils.getXMLValue(xmlFile, XMLValueType.FLOAT, pNAmountKey, "#amount")
                    if sprayTypeName ~= nil and sprayTypeName ~= "" then
                        local modOnLoadData = createModOnLoadData(sprayTypeName)
                        if modOnLoadData ~= nil then
                            if nAmount ~= nil then
                                modOnLoadData.nAmount = nAmount
                            end
                        end
                    end
                end)
            end
        end
        return success
    end
    return false
end
function THPFConfig.loadSprayTypeMappingFromXML(self, xmlFile, xmlKey, customEnv, baseDirectory)
    if THUtils.argIsValid(type(xmlFile) == THValueType.TABLE, "xmlFile", xmlFile)
        and THUtils.argIsValid(type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
        and THUtils.argIsValid(type(customEnv) == THValueType.STRING, "customEnv", customEnv)
        and THUtils.argIsValid(type(baseDirectory) == THValueType.STRING, "baseDirectory", baseDirectory)
    then
        local onLoadValues = self.extendedSprayTypes.onLoadValues
        local success = true
        local sprayTypeMappingKey = xmlKey .. ".sprayTypeMapping"
        if THUtils.hasXMLProperty(xmlFile, sprayTypeMappingKey) then
            xmlFile:iterate(sprayTypeMappingKey .. ".sprayType", function(_, pSprayTypeKey)
                local sprayTypeName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, pSprayTypeKey, "#name")
                local groupName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, pSprayTypeKey, "#group")
                local isLiquid = THUtils.getXMLValue(xmlFile, XMLValueType.BOOL, pSprayTypeKey, "#isLiquid")
                if groupName == nil then
                    groupName = THUtils.getXMLValue(xmlFile, XMLValueType.STRING, pSprayTypeKey, "#parent")
                end
                if sprayTypeName == nil or sprayTypeName == "" then
                    THUtils.xmlErrorMsg(pSprayTypeKey, nil, THMessage.MISSING_XML_KEY, "name")
                elseif groupName == nil or groupName == "" then
                    THUtils.xmlErrorMsg(pSprayTypeKey, nil, THMessage.MISSING_XML_KEY, "group")
                else
                    local sprayTypeId = string.upper(sprayTypeName)
                    local groupId = string.upper(groupName)
                    if groupId == "LIQUIDFERTILIZER" or groupId == "LIQUIDMANURE" then
                        if groupId == "LIQUIDMANURE" then
                            groupId = "MANURE"
                        else
                            groupId = "FERTILIZER"
                        end
                        isLiquid = true -- override
                    end
                    local groupIndex = THPFConfig.SPRAY_TYPE_GROUP:getValue(groupId)
                    if groupIndex == nil then
                        THUtils.xmlErrorMsg(pSprayTypeKey, nil, "Invalid group (%s) for spray type: %q", groupId, sprayTypeId)
                    else
                        local onLoadData = onLoadValues[sprayTypeId]
                        if onLoadData == nil then
                            onLoadData = {}
                            onLoadValues[sprayTypeId] = onLoadData
                        end
                        onLoadData.id = sprayTypeId
                        onLoadData.group = groupIndex
                        if isLiquid ~= nil then
                            onLoadData.isLiquid = isLiquid
                        end
                    end
                end
            end)
        end
        return success
    end
    return false
end
function THPFConfig.finalizeExtendedSprayTypes(self)
    local thMapTypeManager = self.thMapTypeManager
    local sprayTypeManager = self.sprayTypeManager
    if not self.extendedSprayTypes.isModDataLoaded then
        for sprayTypeId, modOnLoadData in pairs(self.extendedSprayTypes.modOnLoadValues) do
            if modOnLoadData.needsRegistration then
                local otherSprayTypeDesc = sprayTypeManager:getSprayTypeByName(sprayTypeId)
                if otherSprayTypeDesc == nil then
                    local sprayGroundType = nil
                    if modOnLoadData.type ~= nil then
                        sprayGroundType = FieldSprayType.getValueByName(modOnLoadData.type)
                        if sprayGroundType == nil then
                            THUtils.errorMsg(false, "Could not find spray ground type: %q", modOnLoadData.type)
                        end
                    end
                    sprayTypeManager:addSprayType(modOnLoadData.id, modOnLoadData.litersPerSecond, modOnLoadData.type, sprayGroundType)
                end
            end
        end
        self.extendedSprayTypes.isModDataLoaded = true
    end
    local sprayTypesArray = sprayTypeManager:getSprayTypes()
    local onLoadValues = self.extendedSprayTypes.onLoadValues
    if sprayTypesArray ~= nil then
        for _, sprayTypeDesc in pairs(sprayTypesArray) do
            if type(sprayTypeDesc.name) == THValueType.STRING and sprayTypeDesc.name ~= ""
                and sprayTypeDesc.fillType ~= nil
            then
                local sprayTypeId = sprayTypeDesc.name
                local thFillTypeData = thMapTypeManager:getFillType(sprayTypeDesc.fillType)
                if self.extendedSprayTypes.byId[sprayTypeId] == nil
                    and thFillTypeData ~= nil and thFillTypeData.index ~= FillType.UNKNOWN
                then
                    local fillTypeIndex = thFillTypeData.index
                    local onLoadData = onLoadValues[thFillTypeData.id]
                    local groupIndex, isLiquid = nil, false
                    if fillTypeIndex == FillType.FERTILIZER or fillTypeIndex == FillType.LIQUIDFERTILIZER then
                        groupIndex = THPFConfig.SPRAY_TYPE_GROUP.FERTILIZER
                    elseif fillTypeIndex == FillType.MANURE or fillTypeIndex == FillType.LIQUIDMANURE then
                        groupIndex = THPFConfig.SPRAY_TYPE_GROUP.MANURE
                    elseif fillTypeIndex == FillType.DIGESTATE then
                        groupIndex = THPFConfig.SPRAY_TYPE_GROUP.DIGESTATE
                    elseif fillTypeIndex == FillType.LIME then
                        groupIndex = THPFConfig.SPRAY_TYPE_GROUP.LIME
                    elseif fillTypeIndex == FillType.HERBICIDE then
                        groupIndex = THPFConfig.SPRAY_TYPE_GROUP.HERBICIDE
                    elseif onLoadData ~= nil then
                        groupIndex = onLoadData.group
                    end
                    if fillTypeIndex == FillType.LIQUIDFERTILIZER
                        or fillTypeIndex == FillType.LIQUIDMANURE
                        or fillTypeIndex == FillType.DIGESTATE
                        or fillTypeIndex == FillType.HERBICIDE
                    then
                        isLiquid = true
                    elseif fillTypeIndex == FillType.FERTILIZER
                        or fillTypeIndex == FillType.MANURE
                        or fillTypeIndex == FillType.LIME
                    then
                        isLiquid = false
                    elseif onLoadData ~= nil then
                        isLiquid = onLoadData.isLiquid
                    end
                    if groupIndex ~= nil then
                        local baseSprayTypeIndex, baseFillTypeIndex = nil, nil
                        if groupIndex == THPFConfig.SPRAY_TYPE_GROUP.FERTILIZER then
                            if isLiquid then
                                baseSprayTypeIndex = SprayType.LIQUIDFERTILIZER
                                baseFillTypeIndex = FillType.LIQUIDFERTILIZER
                            else
                                baseSprayTypeIndex = SprayType.FERTILIZER
                                baseFillTypeIndex = FillType.FERTILIZER
                            end
                        elseif groupIndex == THPFConfig.SPRAY_TYPE_GROUP.MANURE then
                            if isLiquid then
                                baseSprayTypeIndex = SprayType.LIQUIDMANURE
                                baseFillTypeIndex = FillType.LIQUIDMANURE
                            else
                                baseSprayTypeIndex = SprayType.MANURE
                                baseFillTypeIndex = FillType.MANURE
                            end
                        else
                            local sprayTypeGroupId = THPFConfig.SPRAY_TYPE_GROUP:getId(groupIndex)
                            if sprayTypeGroupId ~= nil then
                                baseSprayTypeIndex = SprayType[sprayTypeGroupId]
                                baseFillTypeIndex = FillType[sprayTypeGroupId]
                            end
                        end
                        if baseSprayTypeIndex ~= nil and baseFillTypeIndex ~= nil then
                            local thBaseFillTypeData = self.thMapTypeManager:getFillType(baseFillTypeIndex)
                            local baseSprayTypeDesc = self.sprayTypeManager:getSprayTypeByIndex(baseSprayTypeIndex)
                            if thBaseFillTypeData ~= nil then
                                local baseRate = THUtils.toNumber(baseSprayTypeDesc.litersPerSecond) or 0
                                local usageFactor = 1
                                if baseRate > 0 then
                                    local sprayRate = THUtils.toNumber(sprayTypeDesc.litersPerSecond) or 0
                                    usageFactor = math.max(sprayRate / baseRate, 0.001)
                                end
                                local xtSprayTypeData = {
                                    id = sprayTypeId,
                                    baseId = baseSprayTypeDesc.name,
                                    index = #self.extendedSprayTypes.byIndex + 1,
                                    sprayTypeIndex = sprayTypeDesc.index,
                                    baseSprayTypeIndex = baseSprayTypeIndex,
                                    fillTypeId = thFillTypeData.id,
                                    fillTypeIndex = thFillTypeData.index,
                                    fillTypeTitle = thFillTypeData.title,
                                    baseFillTypeIndex = thBaseFillTypeData.index,
                                    baseFillTypeId = thBaseFillTypeData.id,
                                    group = groupIndex,
                                    isBaseType = thFillTypeData.index == thBaseFillTypeData.index,
                                    isLiquid = isLiquid == true,
                                    usageFactor = usageFactor
                                }
                                function xtSprayTypeData.activateMapping(pSelf, pMapSprayType, pMapFillType)
                                    local vSprayTypeSuccess = false
                                    local vFillTypeSuccess = false
                                    if pMapSprayType == nil or pMapSprayType == true then
                                        if not pSelf.isBaseType then
                                            vSprayTypeSuccess = self.thMapTypeManager:mapSprayTypeConstant(pSelf.baseSprayTypeIndex, pSelf.sprayTypeIndex, true)
                                        end
                                    elseif pMapSprayType ~= false then
                                        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "mapSprayType", pMapSprayType)
                                        vSprayTypeSuccess = false
                                    end
                                    if pMapFillType == nil or pMapFillType == true then
                                        if not pSelf.isBaseType then
                                            vFillTypeSuccess = self.thMapTypeManager:mapFillTypeConstant(pSelf.baseFillTypeIndex, pSelf.fillTypeIndex, true)
                                        end
                                    elseif pMapFillType ~= false then
                                        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "mapFillType", pMapFillType)
                                        vFillTypeSuccess = false
                                    end
                                    return vSprayTypeSuccess == true, vFillTypeSuccess == true
                                end
                                function xtSprayTypeData.resetMapping(pSelf)
                                    local vSprayTypeSuccess = false
                                    local vFillTypeSuccess = false
                                    vSprayTypeSuccess = self.thMapTypeManager:resetSprayTypeConstant(pSelf.baseSprayTypeIndex)
                                    vFillTypeSuccess = self.thMapTypeManager:resetFillTypeConstant(pSelf.baseFillTypeIndex)
                                    return vSprayTypeSuccess, vFillTypeSuccess
                                end
                                self.extendedSprayTypes.byId[xtSprayTypeData.id] = xtSprayTypeData
                                self.extendedSprayTypes.byIndex[xtSprayTypeData.index] = xtSprayTypeData
                                self.extendedSprayTypes.bySprayTypeIndex[xtSprayTypeData.sprayTypeIndex] = xtSprayTypeData
                                local groupSprayTypesArray = self.extendedSprayTypes.byGroup[xtSprayTypeData.group]
                                if groupSprayTypesArray == nil then
                                    groupSprayTypesArray = {}
                                    self.extendedSprayTypes.byGroup[xtSprayTypeData.group] = groupSprayTypesArray
                                end
                                table.insert(groupSprayTypesArray, xtSprayTypeData)
                            end
                        end
                    end
                end
            end
        end
    end
end
function THPFConfig.getExtendedSprayType(self, sprayTypeId, verbose)
    local idType = type(sprayTypeId)
    local xtSprayTypeData = nil
    if THUtils.argIsValid(not verbose or verbose == true, "verbose", verbose) then
        if idType == THValueType.STRING then
            xtSprayTypeData = self.extendedSprayTypes.byId[sprayTypeId:upper()]
        elseif idType == THValueType.NUMBER then
            xtSprayTypeData = self.extendedSprayTypes.bySprayTypeIndex[sprayTypeId]
        elseif sprayTypeId ~= nil then
            verbose = true
        end
        if xtSprayTypeData == nil and verbose then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "sprayTypeId", sprayTypeId)
        end
        return xtSprayTypeData
    end
end
function THPFConfig.getExtendedSprayTypeByFillType(self, fillTypeId, verbose)
    local thFillTypeData = self.thMapTypeManager:getFillType(fillTypeId, verbose)
    if thFillTypeData ~= nil and thFillTypeData.index ~= FillType.UNKNOWN then
        local sprayTypeDesc = self.sprayTypeManager:getSprayTypeByFillTypeIndex(thFillTypeData.index)
        if sprayTypeDesc ~= nil then
            return self:getExtendedSprayType(sprayTypeDesc.name, verbose)
        end
    end
end
function THPFConfig.getExtendedSprayTypes(self, sprayTypeGroupId)
    if sprayTypeGroupId == nil then
        return self.extendedSprayTypes.byIndex, #self.extendedSprayTypes.byIndex
    end
    local groupIdType = type(sprayTypeGroupId)
    local groupIndex = nil
    if groupIdType == THValueType.STRING then
        groupIndex = THPFConfig.SPRAY_TYPE_GROUP:getValue(sprayTypeGroupId:upper())
    elseif groupIdType == THValueType.NUMBER then
        if THPFConfig.SPRAY_TYPE_GROUP:getId(sprayTypeGroupId) then
            groupIndex = sprayTypeGroupId
        end
    end
    if groupIndex ~= nil then
        local groupSprayTypes = self.extendedSprayTypes.byGroup[groupIndex]
        if groupSprayTypes ~= nil then
            return groupSprayTypes, #groupSprayTypes
        end
    end
    THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "sprayTypeGroupId", sprayTypeGroupId)
    return nil, 0
end
function THPFConfig.resetExtendedSprayTypeConstants(self)
    if self.thMapTypeManager:resetSprayTypeConstants()
        and self.thMapTypeManager:resetFillTypeConstants()
    then
        return true
    end
    return false
end
function THPFConfig.addExtendedSprayTypesToFillTypeList(self, sourceNames, sourceCategories, isArray, allowAllSolids, allowAllLiquids)
    if THUtils.argIsValid(not isArray or isArray == true, "isArray", isArray)
        and THUtils.argIsValid(not allowAllSolids or allowAllSolids == true, "allowAllSolids", allowAllSolids)
        and THUtils.argIsValid(not allowAllLiquids or allowAllLiquids == true, "allowAllLiquids", allowAllLiquids)
    then
        local namesType = type(sourceNames)
        local categoriesType = type(sourceCategories)
        local fillTypesArray, fillTypeList = nil, {}
        if namesType == THValueType.TABLE then
            fillTypesArray = {}
            if isArray then
                for _, fillTypeName in ipairs(sourceNames) do
                    local fillTypeId = fillTypeName
                    if type(fillTypeName) == THValueType.STRING then
                        fillTypeId = string.upper(fillTypeName)
                    end
                    if not fillTypeList[fillTypeId] then
                        table.insert(fillTypesArray, fillTypeId)
                        fillTypeList[fillTypeId] = true
                    end
                end
            else
                for fillTypeName, fillTypeValues in pairs(sourceNames) do
                    local fillTypeId = fillTypeName
                    if type(fillTypeName) == THValueType.STRING then
                        fillTypeId = string.upper(fillTypeName)
                    end
                    if fillTypeList[fillTypeId] == nil then
                        table.insert(fillTypesArray, fillTypeId)
                        if sourceCategories == nil then
                            fillTypeList[fillTypeId] = fillTypeValues
                        else
                            fillTypeList[fillTypeId] = true
                        end
                    end
                end
            end
        elseif namesType == THValueType.STRING then
            fillTypesArray = THUtils.splitString(sourceNames, " ", false, true)
            for _, fillTypeId in ipairs(fillTypesArray) do
                fillTypeList[fillTypeId] = true
            end
        elseif sourceNames ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "sourceNames", sourceNames)
            return
        end
        if sourceCategories ~= nil then
            local categoryFillTypes = nil
            if categoriesType == THValueType.TABLE then
                categoryFillTypes = self.fillTypeManager:getFillTypesByCategoryNamesList(sourceCategories)
            elseif categoriesType == THValueType.STRING then
                categoryFillTypes = self.fillTypeManager:getFillTypesByCategoryNames(sourceCategories)
            else
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "sourceCategories", sourceCategories)
                return
            end
            if categoryFillTypes ~= nil then
                if fillTypesArray == nil then
                    fillTypesArray = {}
                end
                for _, categoryFillType in ipairs(categoryFillTypes) do
                    local thFillTypeData = self.thMapTypeManager:getFillType(categoryFillType)
                    if thFillTypeData ~= nil then
                        if fillTypeList[thFillTypeData.id] == nil then
                            table.insert(fillTypesArray, thFillTypeData.id)
                            fillTypeList[thFillTypeData.id] = true
                        end
                    end
                end
            end
        end
        if fillTypesArray == nil or #fillTypesArray <= 0 then
            return
        end
        local hasSprayTypes, hasSolids, hasLiquids = false, false, false
        local arrayIndex = 1
        while true do
            local fillTypeId = fillTypesArray[arrayIndex]
            if fillTypeId == nil then
                break
            end
            local xtSprayTypeData = self:getExtendedSprayTypeByFillType(fillTypeId)
            if xtSprayTypeData ~= nil then
                hasSprayTypes = true
                if xtSprayTypeData.isLiquid then
                    hasLiquids = true
                else
                    hasSolids = true
                end
            end
            arrayIndex = arrayIndex + 1
        end
        if not hasSprayTypes then
            return
        end
        local xtSprayTypesArray = self:getExtendedSprayTypes()
        local hasExtendedSprayTypes = false
        for _, xtSprayTypeData in ipairs(xtSprayTypesArray) do
            if not xtSprayTypeData.isBaseType
                and fillTypeList[xtSprayTypeData.fillTypeId] == nil
            then
                if (hasSolids and not xtSprayTypeData.isLiquid)
                    or (hasLiquids and xtSprayTypeData.isLiquid)
                then
                    local baseValues = fillTypeList[xtSprayTypeData.baseFillTypeId]
                    if baseValues == nil and allowAllSolids then
                        if xtSprayTypeData.group == THPFConfig.SPRAY_TYPE_GROUP.DIGESTATE then
                            if fillTypeList.MANURE
                                and (fillTypeList.FERTILIZER or fillTypeList.LIME)
                            then
                                baseValues = fillTypeList.MANURE
                            end
                        elseif xtSprayTypeData.group == THPFConfig.SPRAY_TYPE_GROUP.HERBICIDE then
                            if fillTypeList.LIME
                                and (fillTypeList.FERTILIZER or fillTypeList.MANURE)
                            then
                                baseValues = fillTypeList.FERTILIZER
                            end
                        end
                    end
                    if baseValues == nil and allowAllLiquids then
                        if xtSprayTypeData.group == THPFConfig.SPRAY_TYPE_GROUP.LIME then
                            if fillTypeList.HERBICIDE
                                and (fillTypeList.LIQUIDFERTILIZER or fillTypeList.LIQUIDMANURE or fillTypeList.DIGESTATE)
                            then
                                baseValues = fillTypeList.LIQUIDFERTILIZER
                            end
                        end
                    end
                    if baseValues ~= nil then
                        table.insert(fillTypesArray, xtSprayTypeData.fillTypeId)
                        fillTypeList[xtSprayTypeData.fillTypeId] = baseValues
                        hasExtendedSprayTypes = true
                    end
                end
            end
        end
        if hasExtendedSprayTypes then
            local fillTypeNames = nil
            for _, fillTypeId in ipairs(fillTypesArray) do
                if fillTypeNames == nil then
                    fillTypeNames = fillTypeId
                else
                    fillTypeNames = fillTypeNames .. " " .. fillTypeId
                end
            end
            if fillTypeNames ~= nil and fillTypeNames ~= "" then
                return fillTypeNames, fillTypeList, fillTypesArray
            end
        end
    end
end
function THPFConfig.inj_getFilename(self, superFunc, filename, filePath, ...)
    if type(filename) == THValueType.STRING and filename ~= ""
        and type(filePath) == THValueType.STRING and filePath ~= ""
    then
        local pfData = self.precisionFarming
        if pfData ~= nil and pfData.isFound then
            THUtils.call(function()
                if filePath == pfData.newBasePath and pfData.newBasePath ~= pfData.oldBasePath then
                    local absFilename = superFunc(filename, filePath)
                    if absFilename == nil or not fileExists(absFilename) then
                        filePath = pfData.oldBasePath
                    end
                end
            end)
        end
    end
    return superFunc(filename, filePath, ...)
end
THUtils.call(initScript)