-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

_G.THValueType = {
    UNKNOWN = "unknown",
    STRING = "string",
    INTEGER = "integer",
    NUMBER = "number",
    BOOLEAN = "boolean",
    TABLE = "table",
    FUNCTION = "function"
}
_G.THGameVersion = 25
if g_minModDescVersion ~= nil then
    if g_minModDescVersion < 90 then
        _G.THGameVersion = 22
    end
end
_G.g_thGlobalEnv = _G
local meta = getmetatable(_G)
if meta ~= nil then
    if type(meta.__index) == THValueType.TABLE then
        _G.g_thGlobalEnv = meta.__index
    else
        printf("ERROR: Could not find global environment")
    end
end
_G.THClassType = {
    OBJECT = "object",
    EVENT = "event"
}
_G.THAxis = {
    X = 1,
    Y = 2,
    Z = 3
}
_G.THColor = {
    TRANSPARENT = { 0, 0, 0, 0 },
    WHITE = { 1, 1, 1, 1 },
    BLACK = { 0, 0, 0, 1 },
    BLUE = { 1, 1, 1, 1 },
    GREEN = { 0.2763, 0.7038, 0.0782, 1 },
    ORANGE = { 0.9900, 0.4640, 0.0010, 1 },
    RED = { 0.9069, 0.0097, 0.0097, 1 },
    BACKGROUND = { 0, 0, 0, 0.75 }
}
THUtils = {
    modName = g_currentModName,
    modPath = g_currentModDirectory
}
THMessage = {
    INTERNAL_ERROR = "An internal error has occurred",
    ARGUMENT_INVALID = "Invalid argument %q (%s) in function call",
    FILE_NOT_FOUND = "File not found: %s",
    INVALID_FILL_TYPE = "Invalid fillType: %s",
    INVALID_FRUIT_TYPE = "Invalid fruitType: %s",
    INVALID_PARENT = "Invalid parent: %s",
    PRELOADING = "Pre-loading: %s",
    LOADING = "Loading: %s",
    LOADING_ERROR = "Failed to load: %s",
    DUPLICATE_ENTRY = "Duplicate %s: %s",
    INVALID_VALUE = "Invalid %s: %s",
    MISSING_XML_KEY = "Missing xml key: %s",
    VALUE_OUT_OF_RANGE = "Value %q (%s) must be between %s and %s",
    VALUE_GREATER = "Value %q (%s) must be greater than %s",
    VALUE_GREATER_EQUAL = "Value %q (%s) must be greater or equal to %s",
    VALUE_LESSER = "Value %q (%s) must be less than %s",
    VALUE_LESSER_EQUAL = "Value %q (%s) must be less than or equal to %s",
    EVENT_ADMIN_ONLY = "Event requires admin privileges",
    EVENT_NO_CLINET = "Event should not run on client machines!",
    EVENT_NO_CLIENT_READ = "Event should not read from client machines!",
    EVENT_NO_CLIENT_WRITE = "Event should not write from client machines!",
    EVENT_NO_SERVER = "Event should not run on the server!",
    EVENT_NO_SERVER_READ = "Event should not read from the server!",
    EVENT_NO_SERVER_WRITE = "Event should not write from the server!"
}
function THUtils.displayMsg(text, ...)
    if text == "" then
        print(text)
    else
        local isError = false
        if type(text) == THValueType.STRING then
            if select("#", ...) > 0 then
                text = string.format(text, ...)
            end
        else
            text = "ERROR: " .. string.format(THMessage.ARGUMENT_INVALID, "text", text)
            isError = true
        end
        local msgText = string.format("** [%s]: %s", THUtils.modName, text)
        print(msgText)
        if isError then
            printCallstack()
        end
    end
end
function THUtils.errorMsg(showStack, text, ...)
    if type(text) == THValueType.STRING and text ~= "" then
        if showStack == nil or showStack == true then
            text = "ERROR: " .. text
        elseif showStack == false then
            text = "WARNING: " .. text
        end
    end
    THUtils.displayMsg(text, ...)
    if showStack == true then
        printCallstack()
    elseif showStack ~= nil and showStack ~= false then
        text = string.format(THMessage.ARGUMENT_INVALID, "showStack", showStack)
        THUtils.displayMsg("ERROR: " .. text)
        printCallstack()
    end
end
function THUtils.msgOnTrue(expression, showStack, text, ...)
    if expression then
        if showStack ~= nil then
            THUtils.errorMsg(showStack, text, ...)
        else
            THUtils.displayMsg(text, ...)
        end
        return true
    end
    return false
end
function THUtils.assert(expression, showStack, text, ...)
    if not expression then
        if showStack ~= nil then
            THUtils.errorMsg(showStack, text, ...)
        else
            THUtils.displayMsg(text, ...)
        end
    end
    return expression
end
function THUtils.assertValue(expression, showStack, yesValue, noValue, text, ...)
    if THUtils.assert(expression, showStack, text, ...) then
        return yesValue
    end
    return noValue
end
function THUtils.argIsValid(expression, argName, argValue)
    if not expression then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, argName, argValue)
        return false
    end
    return true
end
function THUtils.validateArg(expression, argName, argValue, defValue)
    if THUtils.argIsValid(expression, argName, argValue) then
        return argValue
    end
    return defValue
end
function THUtils.xmlErrorMsg(xmlKey, showStack, text, ...)
    text = text or ""
    if THUtils.argIsValid(type(text) == THValueType.STRING, "text", text) then
        local xmlText = "XML Warning: %s"
        if showStack == nil or showStack == true then
            xmlText = "XML Error: %s"
        elseif showStack ~= false then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "showStack", showStack)
        end
        THUtils.displayMsg(xmlText, xmlKey)
        if text ~= nil and text ~= "" then
            THUtils.displayMsg("- " .. text, ...)
        end
        if showStack == true then
            printCallstack()
        end
    end
end
function THUtils.pack(...)
    return { n = select("#", ...), ... }
end
function THUtils.unpack(target, index, endIndex)
    if THUtils.argIsValid(type(target) == THValueType.TABLE, "target", target) then
        if index ~= nil and type(index) ~= "number" then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "index", index)
            index = 1
        end
        if type(endIndex) ~= THValueType.NUMBER then
            if endIndex ~= nil then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "endIndex", endIndex)
            end
            endIndex = target.n
        end
        return unpack(target, index, endIndex)
    end
end
function THUtils.getNoNil(value, replacement)
    if value == nil then
        return replacement
    end
    return value
end
function THUtils.evaluate(expression, yesValue, noValue)
    if expression then
        return yesValue
    end
    return noValue
end
function THUtils.pcall(func, pa,pb,pc,pd,pe,pf,pg,ph,pi,pj,pk,pl,pm,pn,po,pp,pq,pr,ps,pt)
    local function errHandler(errMsg)
        errMsg = THUtils.getNoNil(errMsg, THMessage.INTERNAL_ERROR)
        if type(errMsg) == THValueType.STRING then
            THUtils.errorMsg(true, errMsg)
        else
            printCallstack()
        end
    end
    local function protectedFunc()
        return func(pa,pb,pc,pd,pe,pf,pg,ph,pi,pj,pk,pl,pm,pn,po,pp,pq,pr,ps,pt)
    end
    return xpcall(protectedFunc, errHandler)
end
function THUtils.call(func, ...)
    local function appendFunc(rSuccess, ret1, ...)
        if rSuccess == true then
            return ret1, ...
        end
    end
    return appendFunc(THUtils.pcall(func, ...))
end
function THUtils.getFilename(filename, filePath, verbose, ...)
    if THUtils.argIsValid(filename == nil or type(filename) == THValueType.STRING, "filename", filename)
        and THUtils.argIsValid(filePath == nil or type(filePath) == THValueType.STRING, "filePath", filePath)
        and THUtils.argIsValid(not verbose or verbose == true, "verbose", verbose)
    then
        if filename == nil then
            if verbose == true then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "filename", filename)
            end
        else
            local function appendFunc(rAbsFilename, ...)
                THUtils.call(function()
                    if rAbsFilename == nil or not fileExists(rAbsFilename) then
                        if verbose == true then
                            THUtils.errorMsg(true, THMessage.FILE_NOT_FOUND, rAbsFilename)
                        end
                        if rAbsFilename == nil then
                            rAbsFilename = ""
                        end
                    end
                end)
                return rAbsFilename, ...
            end
            if filePath ~= nil and filePath ~= "" then
                return appendFunc(Utils.getFilename(filename, filePath, ...))
            end
            return appendFunc(Utils.getFilename(filename, nil, ...))
        end
    end
    return ""
end
function THUtils.getFileExists(filename, filePath)
    local absFilename = THUtils.getFilename(filename, filePath)
    if absFilename ~= nil and fileExists(absFilename) then
        return true
    end
    return false
end
function THUtils.toBoolean(value)
    if value == true or value == false then
        return value
    elseif type(value) == THValueType.STRING then
        local upperVal = value:upper()
        if upperVal == "TRUE" then
            return true
        elseif upperVal == "FALSE" then
            return false
        end
    end
end
function THUtils.toString(value, noNil)
    noNil = THUtils.validateArg(not noNil or noNil == true, "noNil", noNil, false)
    if value == nil and noNil == true then
        return value
    end
    return tostring(value)
end
function THUtils.toNumber(value, isInteger)
    isInteger = THUtils.validateArg(not isInteger or isInteger == true, "isInteger", isInteger, false)
    if value ~= nil then
        value = tonumber(value)
        if value ~= nil then
            if isInteger == true then
                value = math.floor(value)
            end
            return value
        end
    end
end
function THUtils.getIsType(value, reqType)
    local valType = type(value)
    if valType == reqType then
        return true
    end
    if reqType == THValueType.INTEGER then
        if valType == THValueType.NUMBER and math.floor(value) == value then
            return true
        end
    else
        local reqTypeVarType = type(reqType)
        if reqTypeVarType == THValueType.TABLE then
            if valType == THValueType.TABLE and value.isa ~= nil and value:isa(reqType) then
                return true
            end
        elseif reqTypeVarType == THValueType.STRING then
            local typeList = THUtils.splitString(reqType, "|", true)
            if typeList ~= nil then
                for otherType in pairs(typeList) do
                    if otherType == valType then
                        return true
                    end
                    if otherType == THValueType.INTEGER then
                        if valType == THValueType.NUMBER and math.floor(value) == value then
                            return true
                        end
                    end
                end
            end
        else
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "reqType", reqType)
        end
    end
    return false
end
function THUtils.convertValue(value, valueType)
    local newValue = nil
    if valueType == nil then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "valueType", valueType)
    elseif valueType == THValueType.STRING then
        newValue = tostring(value)
    elseif valueType == THValueType.INTEGER then
        newValue = THUtils.toNumber(value, true)
    elseif valueType == THValueType.NUMBER then
        newValue = THUtils.toNumber(value)
    elseif valueType == THValueType.BOOLEAN then
        newValue = THUtils.toBoolean(value)
    else
        THUtils.errorMsg(false, "Values of type %q cannot be converted", valueType)
    end
    return newValue
end
function THUtils.clamp(value, minVal, maxVal)
    if THUtils.argIsValid(type(value) == THValueType.NUMBER, "value", value)
        and THUtils.argIsValid(type(minVal) == THValueType.NUMBER, "minVal", minVal)
        and THUtils.argIsValid(type(maxVal) == THValueType.NUMBER, "maxVal", maxVal)
    then
        if minVal > maxVal then
            local oldMinVal = minVal
            minVal = maxVal
            maxVal = oldMinVal
        end
        if THGameVersion == 22 then
            return MathUtil.clamp(value, minVal, maxVal)
        else
            return math.clamp(value, minVal, maxVal)
        end
    end
    return value
end
function THUtils.floor(value, precision)
    value = THUtils.validateArg(type(value) == THValueType.NUMBER, "value", value, 0)
    if type(precision) ~= "number" or precision < 0 then
        if precision ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "precision", precision)
        end
        precision = 0
    else
        precision = math.floor(precision)
    end
    local factor = 10 ^ precision
    return math.floor(value * factor) / factor
end
function THUtils.ceil(value, precision)
    value = THUtils.validateArg(type(value) == THValueType.NUMBER, "value", value, 0)
    if type(precision) ~= "number" or precision < 0 then
        if precision ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "precision", precision)
        end
        precision = 0
    else
        precision = math.floor(precision)
    end
    local factor = 10 ^ precision
    return math.ceil(value * factor) / factor
end
function THUtils.round(value, precision)
    value = THUtils.validateArg(type(value) == THValueType.NUMBER, "value", value, 0)
    if type(precision) ~= "number" or precision < 0 then
        if precision ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "precision", precision)
        end
        precision = 0
    else
        precision = math.floor(precision)
    end
    local factor = 10 ^ precision
    return math.floor((value * factor) + 0.5) / factor
end
function THUtils.getVector2Distance(sx,sz, ex,ez)
    if ex == nil and ez == nil then
        ex = 0
        ez = 0
    end
    if THUtils.argIsValid(type(sx) == THValueType.NUMBER, "sx", sx)
        and THUtils.argIsValid(type(sz) == THValueType.NUMBER, "sz", sz)
        and THUtils.argIsValid(type(ex) == THValueType.NUMBER, "ex", ex)
        and THUtils.argIsValid(type(ez) == THValueType.NUMBER, "ez", ez)
    then
        return math.sqrt(((ex - sx) ^ 2) + ((ez - sz) ^ 2))
    end
    return 0
end
function THUtils.getVector3Distance(sx,sy,sz, ex,ey,ez)
    if ex == nil and ey == nil and ez == nil then
        ex = 0
        ey = 0
        ez = 0
    end
    if THUtils.argIsValid(type(sx) == THValueType.NUMBER, "sx", sx)
        and THUtils.argIsValid(type(sy) == THValueType.NUMBER, "sy", sy)
        and THUtils.argIsValid(type(sz) == THValueType.NUMBER, "sz", sz)
        and THUtils.argIsValid(type(ex) == THValueType.NUMBER, "ex", ex)
        and THUtils.argIsValid(type(ey) == THValueType.NUMBER, "ey", ey)
        and THUtils.argIsValid(type(ez) == THValueType.NUMBER, "ez", ez)
    then
        return math.sqrt(((ex - sx) ^ 2) + ((ey - sy) ^ 2) + ((ez - sz) ^ 2))
    end
    return 0
end
function THUtils.getVector2AreaByLine(sx, sz, ex, ez, width)
    if THUtils.argIsValid(type(sx) == THValueType.NUMBER, "sx", sx)
        and THUtils.argIsValid(type(sz) == THValueType.NUMBER, "sz", sz)
        and THUtils.argIsValid(type(ex) == THValueType.NUMBER, "ex", ex)
        and THUtils.argIsValid(type(ez) == THValueType.NUMBER, "ez", ez)
    then
        if type(width) ~= "number" or width < 0 then
            if width ~= nil then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "width", width)
            end
            width = 0
        end
        local length = THUtils.getVector2Distance(sx, sz, ex, ez)
        local mx = (sx + ex) / 2
        local mz = (sz + ez) / 2
        if length == 0 then -- single point
            if width == 0 then -- return single point
                return mx, mz, mx, mz, mx, mz
            end
            local halfWidth = width / 2
            return mx - halfWidth, mz - halfWidth, mx + halfWidth, mz - halfWidth, mx - halfWidth, mz + halfWidth
        end
        local factor = 1 + (width / length)
        local dx = ((ex - sx) / 2) * factor
        local dz = ((ez - sz) / 2) * factor
        sx = mx - dx
        sz = mz - dz
        ex = mx + dx
        ez = mz + dz
        local rdx = dz
        local rdz = -dx
        return sx + rdx, sz + rdz, sx - rdx, sz - rdz, ex + rdx, ez + rdz
    end
end
function THUtils.getFactorInRange(val1, val2, testVal, noClamp)
    if THUtils.argIsValid(type(val1) == THValueType.NUMBER, "val1", val1)
        and THUtils.argIsValid(type(val2) == THValueType.NUMBER, "val2", val2)
        and THUtils.argIsValid(type(testVal) == THValueType.NUMBER, "testVal", testVal)
    then
        if val1 == val2 then
            if testVal < val1 then
                return 0
            else
                return 1
            end
        end
        local factor = (testVal - val1) / (val2 - val1)
        if noClamp ~= true then
            if noClamp ~= nil and noClamp ~= false then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "noClamp", noClamp)
            end
            factor = THUtils.clamp(factor, 0, 1)
        end
        return factor
    end
    return 0
end
function THUtils.getValueInRange(val1, val2, alpha, noClamp)
    if THUtils.argIsValid(type(val1) == THValueType.NUMBER, "val1", val1)
        and THUtils.argIsValid(type(val2) == THValueType.NUMBER, "val2", val2)
        and THUtils.argIsValid(type(alpha) == THValueType.NUMBER, "alpha", alpha)
    then
        local newValue = val1 + ((val2 - val1) * alpha)
        if noClamp ~= true then
            if noClamp ~= nil and noClamp ~= false then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "noClamp", noClamp)
            end
            local minVal = math.min(val1, val2)
            local maxVal = math.max(val1, val2)
            newValue = THUtils.clamp(newValue, minVal, maxVal)
        end
        return newValue
    end
    if type(val1) == THValueType.NUMBER then
        return val1
    elseif type(val2) == THValueType.NUMBER then
        return val2
    end
    return 0
end
function THUtils.splitString(str, pattern, isList, toUpper)
    if type(pattern) ~= THValueType.STRING or pattern == "" then
        if pattern ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "pattern", pattern)
        end
        pattern = " "
    end
    toUpper = THUtils.validateArg(not toUpper or toUpper == true, "toUpper", toUpper, false)
    local splitTable = {}
    if str == nil or str == "" then
        return splitTable
    end
    if THUtils.argIsValid(type(str) == THValueType.STRING, "str", str)
        and THUtils.argIsValid(not isList or isList == true, "isList", isList)
    then
        local valToAdd, valLength = nil, string.len(str)
        local strPos = 1
        while true do
            if strPos <= 0 or strPos > valLength then
                break
            end
            local strStart, strEnd = string.find(str, pattern, strPos, true)
            if strStart == nil or strEnd == nil then
                valToAdd = string.sub(str, strPos)
                strPos = valLength + 1
            else
                if strEnd <= 1 then
                    valToAdd = nil
                else
                    valToAdd = string.sub(str, strPos, strEnd - 1)
                end
                strPos = strEnd + 1
            end
            if valToAdd ~= nil and valToAdd ~= "" and valToAdd ~= pattern then
                if toUpper and type(valToAdd) == THValueType.STRING then
                    valToAdd = string.upper(valToAdd)
                end
                if isList == true then
                    splitTable[valToAdd] = true
                else
                    table.insert(splitTable, valToAdd)
                end
            end
        end
    end
    return splitTable
end
function THUtils.properCase(value, noSpace)
    local strValue = tostring(value)
    if THUtils.argIsValid(not noSpace or noSpace == true, "noSpace", noSpace) then
        strValue = string.lower(strValue)
        strValue = string.gsub(strValue, "^%a", string.upper)
        strValue = string.gsub(strValue, "[_ ]%a", string.upper)
        if noSpace then
            strValue = string.gsub(strValue, "[_ ]", "")
        else
            strValue = string.gsub(strValue, "_", "")
        end
    end
    return strValue
end
function THUtils.camelCase(value, noSpace)
    local strValue = tostring(value)
    if THUtils.argIsValid(not noSpace or noSpace == true, "noSpace", noSpace) then
        strValue = string.lower(strValue)
        strValue = string.gsub(strValue, "_%a", string.upper)
        if noSpace then
            strValue = string.gsub(strValue, "[_ ]", "")
        else
            strValue = string.gsub(strValue, "_", "")
        end
    end
    return strValue
end
function THUtils.snakeCase(value, toUpper, noSpace)
    local strValue = tostring(value)
    if THUtils.argIsValid(not toUpper or toUpper == true, "toUpper", toUpper)
        and THUtils.argIsValid(not noSpace or noSpace == true, "noSpace", noSpace)
    then
        strValue = string.gsub(strValue, "(%l)(%u)", "%1_%2")
        if toUpper then
            strValue = string.upper(strValue)
        else
            strValue = string.lower(strValue)
        end
        if noSpace then
            strValue = string.gsub(strValue, " ", "_")
        end
    end
    return strValue
end
function THUtils.validateId(id)
    if id ~= nil and id ~= "" and type(id) == THValueType.STRING
        and not string.find(id, "[^A-Za-z0-9_]")
    then
        return true
    end
    return false
end
function THUtils.formatPercent(value, precision)
    value = THUtils.validateArg(type(value) == THValueType.NUMBER, "value", value, 0)
    if type(precision) ~= "number" or precision < 0 then
        if precision ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "precision", precision)
        end
        precision = 0
    else
        precision = math.floor(precision)
    end
    local percentValue = THUtils.round(value * 100, precision)
    local percentText = string.format("%0." .. tostring(precision) .. "f%%", percentValue)
    return percentText
end
function THUtils.clearTable(target)
    if THUtils.argIsValid(type(target) == THValueType.TABLE, "target", target) then
        for key in pairs(target) do
            target[key] = nil
        end
    end
    return target
end
function THUtils.sortTable(...)
    return table.sort(...)
end
function THUtils.getTableValue(target, targetPath, pattern)
    if target == nil then
        target = g_thGlobalEnv
    elseif type(target) ~= "table" then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "target", target)
        return
    end
    if type(pattern) ~= THValueType.STRING or pattern == "" then
        if pattern ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "pattern", pattern)
        end
        pattern = "."
    end
    if targetPath == nil or targetPath == "" then
        return target
    end
    if THUtils.argIsValid(type(targetPath) == THValueType.STRING, "targetPath", targetPath) then
        if not string.find(targetPath, pattern, nil, true) then
            return target[targetPath]
        end
        local tableKeys = THUtils.splitString(targetPath, pattern)
        local numEntries = tableKeys ~= nil and #tableKeys
        if numEntries > 0 then
            local finalValue = target
            for entryIndex = 1, numEntries do
                local entry = tableKeys[entryIndex]
                if entryIndex < numEntries and type(finalValue) ~= "table" then
                    THUtils.errorMsg(true, "Invalid table: %s", entry)
                    return
                end
                if finalValue[entry] ~= nil then
                    finalValue = finalValue[entry]
                else
                    local otherEntry = THUtils.toBoolean(entry)
                    if otherEntry ~= nil and finalValue[otherEntry] ~= nil then
                        finalValue = finalValue[otherEntry]
                    else
                        otherEntry = THUtils.toNumber(entry)
                        if otherEntry ~= nil and finalValue[otherEntry] ~= nil then
                            finalValue = finalValue[otherEntry]
                        end
                    end
                end
            end
            return finalValue
        end
    end
end
function THUtils.copyTable(target, depth, copyMeta)
    local newTable = {}
    if THUtils.argIsValid(type(target) == THValueType.TABLE, "target", target) then
        if type(depth) ~= "number" or depth < 0 then
            if depth ~= nil then
                THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "depth", depth)
            end
            depth = 0
        else
            depth = math.floor(depth)
        end
        for key, val in pairs(target) do
            if type(val) == THValueType.TABLE and depth > 0 then
                newTable[key] = THUtils.copyTable(val, depth - 1)
            else
                newTable[key] = val
            end
        end
        if copyMeta == true then
            local oldMt = getmetatable(target)
            if oldMt ~= nil then
                setmetatable(newTable, oldMt)
            end
        elseif copyMeta ~= nil and copyMeta ~= false then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "copyMeta", copyMeta)
        end
    end
    return newTable
end
function THUtils.copyFunctions(tgtTable, srcTable, allowHooks)
    if THUtils.argIsValid(type(tgtTable) == THValueType.TABLE, "tgtTable", tgtTable)
        and THUtils.argIsValid(type(srcTable) == THValueType.TABLE, "srcTable", srcTable)
        and THUtils.argIsValid(not allowHooks or allowHooks == true, "allowHooks", allowHooks)
    then
        if srcTable == tgtTable then
            return true
        end
        for key, value in pairs(srcTable) do
            if type(value) == THValueType.FUNCTION then
                if tgtTable[key] == nil then
                    local allowAdd = true
                    if not allowHooks then
                        local strKey = tostring(key)
                        if string.sub(strKey, 1, 4) == "inj_"
                            or string.sub(strKey, 1, 5) == "hook_"
                        then
                            allowAdd = false
                        end
                    end
                    if allowAdd then
                        tgtTable[key] = value
                    end
                elseif tgtTable[key] ~= value then
                    THUtils.errorMsg(false, "Function %q already exists", key)
                end
            end
        end
        return true
    end
    return false
end
function THUtils.createSubTable(parent, tableKey, clear, useRaw)
    if THUtils.argIsValid(type(parent) == THValueType.TABLE, "parent", parent)
        and THUtils.argIsValid(tableKey ~= nil, "tableKey", tableKey)
        and THUtils.argIsValid(not clear or clear == true, "clear", clear)
        and THUtils.argIsValid(not useRaw or useRaw == true, "useRaw", useRaw)
    then
        local newTable = nil
        if useRaw == true then
            newTable = rawget(parent, tableKey)
        else
            newTable = parent[tableKey]
        end
        if newTable == nil then
            newTable = {}
            if useRaw == true then
                rawset(parent, tableKey, newTable)
            else
                parent[tableKey] = newTable
            end
        elseif clear then
            THUtils.clearTable(newTable)
        end
        return newTable
    end
end
function THUtils.getTableLength(target)
    if THUtils.argIsValid(type(target) == THValueType.TABLE, "target", target) then
        local numEntries = 0
        if target ~= nil then
            for _ in pairs(target) do
                numEntries = numEntries + 1
            end
        end
        return numEntries
    end
    return 0
end
function THUtils.getDataTable(target, dataKey)
    if type(target) == THValueType.TABLE then
        dataKey = THUtils.getNoNil(dataKey, target)
        local dataTable = rawget(target, dataKey)
        return dataTable, target
    end
    if target ~= nil then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "target", target)
    end
end
function THUtils.createDataTable(target, dataKey, customMt, ...)
    if THUtils.argIsValid(type(target) == THValueType.TABLE, "target", target)
        and THUtils.argIsValid(customMt == nil or type(customMt) == THValueType.TABLE, "customMt", customMt)
    then
        dataKey = THUtils.getNoNil(dataKey, target)
        local dataTable = THUtils.getDataTable(target, dataKey)
        if dataTable == nil then
            dataTable = { parent = target }
            rawset(target, dataKey, dataTable)
        end
        if customMt ~= nil then
            local currentMt = getmetatable(dataTable)
            if currentMt == nil then
                setmetatable(dataTable, customMt)
            elseif currentMt ~= customMt then
                THUtils.errorMsg(true, "Cannot change data metatable")
            end
        end
        return dataTable
    end
end
function THUtils.createEnumTable(target)
    if THUtils.argIsValid(type(target) == THValueType.TABLE, "target", target) then
        if target.getValues ~= nil then
            THUtils.errorMsg(true, "Enum table already created")
            return
        end
        local idToValue = {}
        local valueToId = {}
        for id, value in pairs(target) do
            if THUtils.getIsType(id, "string|integer")
                and THUtils.getIsType(value, "integer|number|string|boolean")
                and idToValue[id] == nil
            then
                if valueToId[value] == nil then
                    idToValue[id] = value
                    valueToId[value] = id
                else
                    THUtils.errorMsg(true, THMessage.DUPLICATE_ENTRY, "value", value)
                end
            end
        end
        local enumTable = target
        enumTable.getId = function(pSelf, pValue, pAllowReverse)
            if THUtils.argIsValid(not pAllowReverse or pAllowReverse == true, "allowReverse", pAllowReverse)
                and pValue ~= nil
            then
                local id = valueToId[pValue]
                if id == nil then
                    if type(pValue) == THValueType.STRING then
                        local upperValue = string.upper(pValue)
                        id = valueToId[upperValue]
                        if pAllowReverse
                            and id == nil and idToValue[upperValue] ~= nil
                        then
                            id = upperValue
                        end
                    end
                    if pAllowReverse
                        and id == nil and idToValue[pValue] ~= nil
                    then
                        id = pValue
                    end
                end
                return id
            end
        end
        enumTable.getValue = function(pSelf, pId, pAllowReverse)
            if THUtils.argIsValid(not pAllowReverse or pAllowReverse == true, "isStrict", pAllowReverse)
                and pId ~= nil
            then
                local value = idToValue[pId]
                if value == nil then
                    if type(pId) == THValueType.STRING then
                        local upperValue = string.upper(pId)
                        value = idToValue[upperValue]
                        if pAllowReverse
                            and value == nil and valueToId[upperValue] ~= nil
                        then
                            value = upperValue
                        end
                    end
                    if pAllowReverse
                        and value == nil and valueToId[pId] ~= nil
                    then
                        value = pId
                    end
                end
                return value
            end
        end
        enumTable.getValues = function(pSelf)
            return idToValue
        end
    end
end
function THUtils.registerValue(target, key, value)
    if THUtils.argIsValid(type(target) == THValueType.TABLE, "target", target)
        and THUtils.argIsValid(key ~= nil, "key", key)
        and THUtils.argIsValid(value ~= nil, "value", value)
    then
        local oldValue = target[key]
        if oldValue == nil then
            target[key] = value
        elseif value ~= oldValue then
            THUtils.errorMsg(true, "Table value %q already exists", key)
        end
    end
end
function THUtils.makeSelfCallback(target, targetFunc)
    if target == nil then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "target", target)
    else
        local callbackFunc = function(...)
            return targetFunc(target, ...)
        end
        return callbackFunc
    end
end
function THUtils.addTask(isProtected, taskFunc)
    local taskManager = g_asyncTaskManager
    if THUtils.argIsValid(not isProtected or isProtected == true, "isProtected", isProtected)
        and THUtils.argIsValid(type(taskFunc) == THValueType.FUNCTION, "taskFunc", taskFunc)
    then
        if taskManager == nil then
            THUtils.errorMsg(true, "Could not find task manager")
        else
            if isProtected == true then
                taskManager:addTask(function()
                    THUtils.pcall(taskFunc)
                end)
            else
                taskManager:addTask(taskFunc)
            end
        end
    end
end
function THUtils.addSubTask(isProtected, taskFunc)
    local taskManager = g_asyncTaskManager
    if THUtils.argIsValid(not isProtected or isProtected == true, "isProtected", isProtected)
        and THUtils.argIsValid(type(taskFunc) == THValueType.FUNCTION, "taskFunc", taskFunc)
    then
        if taskManager == nil then
            THUtils.errorMsg(true, "Could not find task manager")
        else
            if isProtected == true then
                taskManager:addSubtask(function()
                    THUtils.pcall(taskFunc)
                end)
            else
                taskManager:addSubtask(taskFunc)
            end
        end
    end
end
function THUtils.addCallbackToArray(callbackArray, callbackTarget, callbackFunc, ...)
    if THUtils.argIsValid(type(callbackArray) == THValueType.TABLE, "callbackArray", callbackArray)
        and THUtils.argIsValid(type(callbackFunc) == THValueType.FUNCTION, "callbackFunc", callbackFunc)
    then
        local numEntries = #callbackArray
        local callbackData = nil
        local isCallbackFound = false
        if numEntries > 0 then
            for entryIndex = 1, numEntries do
                callbackData = callbackArray[entryIndex]
                if callbackData.func == callbackFunc then
                    isCallbackFound = true
                    break
                end
            end
        end
        if not isCallbackFound then
            callbackData = {
                func = callbackFunc,
                target = callbackTarget,
                args = THUtils.pack(...)
            }
            table.insert(callbackArray, callbackData)
        end
        return true, callbackFunc
    end
    return false
end
function THUtils.removeCallbackFromArray(callbackArray, callbackFunc)
    if THUtils.argIsValid(type(callbackArray) == THValueType.TABLE, "callbackArray", callbackArray) then
        if type(callbackFunc) == THValueType.FUNCTION then
            local entryIndex = 1
            local callbackData = nil
            local isCallbackFound = false
            while true do
                callbackData = callbackArray[entryIndex]
                if callbackData == nil then
                    break
                end
                if callbackData.func == callbackFunc then
                    table.remove(callbackArray, entryIndex)
                    isCallbackFound = true
                else
                    entryIndex = entryIndex + 1
                end
            end
            return isCallbackFound
        elseif callbackFunc ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "callbackFunc", callbackFunc)
        end
    end
    return false
end
function THUtils.raiseArrayCallbacks(callbackArray, ...)
    if THUtils.argIsValid(type(callbackArray) == THValueType.TABLE, "callbackArray", callbackArray) then
        local callbackArgs = THUtils.pack(...)
        for entryIndex = 1, #callbackArray do
            local callbackData = callbackArray[entryIndex]
            local hasCallbackArgs = callbackData.args.n > 0
            local function taskFunc()
                if callbackData.target ~= nil then
                    if hasCallbackArgs then
                        callbackData.func(callbackData.target, THUtils.unpack(callbackData.args), THUtils.unpack(callbackArgs))
                    else
                        callbackData.func(callbackData.target, THUtils.unpack(callbackArgs))
                    end
                else
                    if hasCallbackArgs then
                        callbackData.func(THUtils.unpack(callbackData.args), THUtils.unpack(callbackArgs))
                    else
                        callbackData.func(THUtils.unpack(callbackArgs))
                    end
                end
            end
            if entryIndex == 1 then
                THUtils.addTask(true, taskFunc)
            else
                THUtils.addSubTask(true, taskFunc)
            end
        end
    end
end
function THUtils.registerFunction(target, funcName, newFunc)
    if THUtils.argIsValid(type(target) == THValueType.TABLE, "target", target)
        and THUtils.argIsValid(type(funcName) == THValueType.STRING, "funcName", funcName)
        and THUtils.argIsValid(type(newFunc) == THValueType.FUNCTION, "newFunc", newFunc)
    then
        local oldFunc = rawget(target, funcName)
        if oldFunc ~= nil then
            THUtils.errorMsg(true, "Function %q already registered", funcName)
            return oldFunc
        end
        local callbackFunc = function(...)
            return newFunc(...)
        end
        rawset(target, funcName, callbackFunc)
        return callbackFunc
    end
end
function THUtils.setFunctionHook(srcTable, srcFuncName, allowCreate, useSelf, extraArg, tgtFunc)
    if srcTable == nil then
        srcTable = _G
    else
        local srcTableName = srcTable
        if type(srcTable) == THValueType.STRING then
            srcTable = THUtils.getTableValue(nil, srcTableName)
        end
        if type(srcTable) ~= "table" then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "srcTable", srcTableName)
            return false
        end
    end
    if THUtils.argIsValid(type(srcFuncName) == THValueType.STRING, "srcFuncName", srcFuncName)
        and THUtils.argIsValid(type(tgtFunc) == THValueType.FUNCTION, "tgtFunc", tgtFunc)
        and THUtils.argIsValid(not allowCreate or allowCreate == true, "allowCreate", allowCreate)
    then
        local srcFunc = srcTable[srcFuncName]
        if type(srcFunc) ~= "function" then
            if srcFunc == nil and allowCreate == true then
                srcFunc = function() end
            else
                THUtils.errorMsg(true, "Invalid source function: %s", srcFuncName)
                return false
            end
        end
        local function callbackFunc(p1, ...)
            if useSelf == true then
                if extraArg ~= nil then
                    return tgtFunc(p1, srcFunc, extraArg, ...)
                else
                    return tgtFunc(p1, srcFunc, ...)
                end
            else
                if extraArg ~= nil then
                    return tgtFunc(extraArg, srcFunc, p1, ...)
                else
                    return tgtFunc(srcFunc, p1, ...)
                end
            end
        end
        rawset(srcTable, srcFuncName, callbackFunc)
        return true, callbackFunc, srcTable
    end
    return false
end
function THUtils.getIsMasterUser()
    if g_currentMission ~= nil then
        return g_currentMission.isMasterUser == true
    end
    return false
end
function THUtils.getModEnvironment(modName)
    if modName == nil then
        return _G
    end
    if THUtils.argIsValid(type(modName) == THValueType.STRING, "modName", modName) then
        local modEnv = g_thGlobalEnv[modName]
        if modEnv ~= nil and modEnv._G ~= nil then
            return modEnv._G
        end
    end
end
function THUtils.getClassObject(className, modName)
    local classObject = nil
    if modName == nil then
        classObject = THUtils.getTableValue(g_thGlobalEnv, className)
    else
        local modEnv = THUtils.getModEnvironment(modName)
        if modEnv ~= nil then
            classObject = THUtils.getTableValue(modEnv, className)
        end
    end
    return classObject
end
function THUtils.createClass(baseClass, parentClass, ...)
    if THUtils.argIsValid(type(baseClass) == THValueType.TABLE, "baseClass", baseClass)
        and THUtils.argIsValid(parentClass == nil or type(parentClass) == THValueType.TABLE, "parentClass", parentClass)
    then
        return Class(baseClass, parentClass, ...)
    end
    return {}
end
function THUtils.initClass(classTable, className, classType, ...)
    if THUtils.argIsValid(type(classTable) == THValueType.TABLE, "classTable", classTable)
        and THUtils.argIsValid(type(className) == THValueType.STRING and className ~= "", "className", className)
    then
        if classType == THClassType.OBJECT then
            return InitObjectClass(classTable, className, ...)
        elseif classType == THClassType.EVENT then
            return InitEventClass(classTable, className, ...)
        else
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "classType", classType)
        end
    end
end
function THUtils.callSuperClass(target, funcName, useSelf, ...)
    if THUtils.argIsValid(funcName ~= nil, "funcName", funcName)
        and THUtils.argIsValid(not useSelf or useSelf == true, "useSelf", useSelf)
    then
        if type(target) == THValueType.TABLE and type(target.superClass) == THValueType.FUNCTION
            and type(target[funcName]) == THValueType.FUNCTION
        then
            local superClass = target:superClass()
            if type(superClass) == THValueType.TABLE
                and type(superClass[funcName]) == THValueType.FUNCTION
            then
                if useSelf then
                    return THUtils.call(superClass[funcName], target, ...)
                else
                    return THUtils.call(superClass[funcName], ...)
                end
            end
            return true
        else
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "target", target)
        end
    end
end
function THUtils.addInitSchemaFunction(target, xmlName, initFunc)
    if THUtils.argIsValid(type(target) == THValueType.TABLE, "target", target)
        and THUtils.argIsValid(type(xmlName) == THValueType.STRING, "xmlName", xmlName)
        and THUtils.argIsValid(type(initFunc) == THValueType.FUNCTION, "initFunc", initFunc)
    then
        local xmlManager = g_xmlManager
        if xmlManager ~= nil then
            xmlManager:addCreateSchemaFunction(function()
                target.xmlSchema = XMLSchema.new(xmlName)
            end)
            xmlManager:addInitSchemaFunction(initFunc)
        end
    end
end
function THUtils.getXMLRootKey(xmlFile)
    local xmlFileType = type(xmlFile)
    local xmlRootKey = nil
    if xmlFileType == THValueType.TABLE then
        xmlRootKey = xmlFile:getRootName()
    elseif xmlFileType == THValueType.NUMBER then
        xmlRootKey = getXMLRootName(xmlFile)
    else
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "xmlFile", xmlFile)
    end
    return xmlRootKey
end
function THUtils.loadXMLFile(xmlName, filename, filePath, xmlSchema)
    local xmlFile = nil
    local xmlFilename = THUtils.getFilename(filename, filePath)
    if THUtils.argIsValid(type(xmlName) == THValueType.STRING, "xmlName", xmlName)
        and xmlFilename ~= nil and xmlFilename ~= ""
    then
        if xmlSchema == false then
            if fileExists(xmlFilename) then
                xmlFile = loadXMLFile(xmlName, xmlFilename)
                if xmlFile ~= nil and xmlFile <= 0 then
                    THUtils.errorMsg(true, "Invalid xml file (%s)", xmlName)
                    return
                end
            end
        elseif xmlSchema == nil or xmlSchema == true then
            xmlFile = XMLFile.loadIfExists(xmlName, xmlFilename)
        elseif type(xmlSchema) == THValueType.TABLE then
            xmlFile = XMLFile.loadIfExists(xmlName, xmlFilename, xmlSchema)
        else
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "xmlSchema", xmlSchema)
        end
        return xmlFile, xmlFilename
    end
end
function THUtils.deleteXMLFile(xmlFile)
    local xmlFileType = type(xmlFile)
    if xmlFileType == THValueType.TABLE then
        xmlFile:delete()
    elseif xmlFileType == THValueType.NUMBER then
        delete(xmlFile)
    else
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "xmlFile", xmlFile)
    end
end
function THUtils.registerXMLPath(xmlSchema, xmlValueType, xmlPath, xmlKey, ...)
    if THUtils.argIsValid(type(xmlSchema) == THValueType.TABLE and xmlSchema.register ~= nil, "xmlSchema", xmlSchema)
        and THUtils.argIsValid(type(xmlPath) == THValueType.STRING, "xmlPath", xmlPath)
        and THUtils.argIsValid(xmlKey == nil or type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
    then
        local absXMLPath = xmlPath
        if xmlKey ~= nil then
            absXMLPath = xmlPath .. xmlKey
        end
        return xmlSchema:register(xmlValueType, absXMLPath, ...)
    end
end
function THUtils.hasXMLProperty(xmlFile, ...)
    if type(xmlFile) == THValueType.TABLE
        and type(xmlFile.hasProperty) == THValueType.FUNCTION
    then
        return xmlFile:hasProperty(...)
    elseif THUtils.getIsType(xmlFile, THValueType.INTEGER) and xmlFile > 0 then
        return hasXMLProperty(xmlFile, ...)
    else
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "xmlFile", xmlFile)
    end
    return false
end
function THUtils.removeXMLProperty(xmlFile, xmlKey, ...)
    if THUtils.argIsValid(type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey) then
        if type(xmlFile) == THValueType.TABLE then
            if type(xmlFile.removeProperty) == THValueType.FUNCTION then
                return xmlFile:removeProperty(xmlKey, ...)
            end
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "xmlFile", xmlFile)
        elseif THUtils.getIsType(xmlFile, THValueType.INTEGER) and xmlFile > 0 then
            return removeXMLProperty(xmlFile, xmlKey, ...)
        else
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "xmlFile", xmlFile)
        end
    end
end
function THUtils.getXMLValueData(xmlFile, xmlValueType)
    local xmlFileType = type(xmlFile)
    local getValueFunc = nil
    local setValueFunc = nil
    local hasDefault = xmlFileType == THValueType.TABLE
    if THUtils.argIsValid(xmlFileType == THValueType.TABLE or xmlFileType == THValueType.NUMBER, "xmlFile", xmlFile) then
        if xmlValueType == nil then
            if xmlFileType == THValueType.TABLE then
                getValueFunc = xmlFile.getValue
                setValueFunc = xmlFile.setValue
            end
        elseif xmlValueType == XMLValueType.STRING then
            if xmlFileType == THValueType.TABLE then
                getValueFunc = xmlFile.getString
                setValueFunc = xmlFile.setString
            else
                getValueFunc = getXMLString
                setValueFunc = setXMLString
            end
        elseif xmlValueType == XMLValueType.FLOAT then
            if xmlFileType == THValueType.TABLE then
                getValueFunc = xmlFile.getFloat
                setValueFunc = xmlFile.setFloat
            else
                getValueFunc = getXMLFloat
                setValueFunc = setXMLFloat
            end
        elseif xmlValueType == XMLValueType.INT then
            if xmlFileType == THValueType.TABLE then
                getValueFunc = xmlFile.getInt
                setValueFunc = xmlFile.setInt
            else
                getValueFunc = getXMLInt
                setValueFunc = setXMLInt
            end
        elseif xmlValueType == XMLValueType.BOOL then
            if xmlFileType == THValueType.TABLE then
                getValueFunc = xmlFile.getBool
                setValueFunc = xmlFile.setBool
            else
                getValueFunc = getXMLBool
                setValueFunc = setXMLBool
            end
        end
    end
    if getValueFunc == nil then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "xmlValueType", xmlValueType)
    end
    return getValueFunc, setValueFunc, hasDefault
end
function THUtils.getXMLValue(xmlFile, xmlValueType, xmlPath, xmlKey, defaultValue, ...)
    if THUtils.argIsValid(type(xmlPath) == THValueType.STRING, "xmlPath", xmlPath)
        and THUtils.argIsValid(xmlKey == nil or type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
    then
        local getValueFunc, _, hasDefault = THUtils.getXMLValueData(xmlFile, xmlValueType)
        local absXMLKey = xmlPath
        if xmlKey ~= nil then
            absXMLKey = xmlPath .. xmlKey
        end
        if getValueFunc ~= nil then
            local function appendFunc(rValue, ...)
                if rValue == nil and not hasDefault then
                    rValue = defaultValue
                end
                return rValue, ...
            end
            if hasDefault then
                return appendFunc(getValueFunc(xmlFile, absXMLKey, defaultValue, ...))
            end
            return appendFunc(getValueFunc(xmlFile, absXMLKey, ...))
        end
    end
    return defaultValue
end
function THUtils.setXMLValue(xmlFile, xmlValueType, xmlPath, xmlKey, ...)
    if THUtils.argIsValid(type(xmlPath) == THValueType.STRING, "xmlPath", xmlPath)
        and THUtils.argIsValid(xmlKey == nil or type(xmlKey) == THValueType.STRING, "xmlKey", xmlKey)
    then
        local _, setValueFunc = THUtils.getXMLValueData(xmlFile, xmlValueType)
        local absXMLKey = xmlPath
        if xmlKey ~= nil then
            absXMLKey = xmlPath .. xmlKey
        end
        if setValueFunc ~= nil then
            return setValueFunc(xmlFile, absXMLKey, ...)
        end
    end
end
function THUtils.getNormalizedValues(uiScale, x, y)
    local normX, normY = 0, 0
    if THUtils.argIsValid(type(uiScale) == THValueType.NUMBER, "uiScale", uiScale) then
        if type(x) == THValueType.NUMBER then
            normX = x * uiScale * g_aspectScaleX / g_referenceScreenWidth
        elseif x ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "x", x)
        end
        if type(y) == THValueType.NUMBER then
            normY = y * uiScale * g_aspectScaleY / g_referenceScreenHeight
        elseif y ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "y", y)
        end
    end
    return normX, normY
end
function THUtils.getElementByProfileName(baseElement, profileName, depth)
    if type(depth) ~= "number" or depth < 0 then
        if depth ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "depth", depth)
        end
        depth = 0
    else
        depth = math.floor(depth)
    end
    if THUtils.argIsValid(type(baseElement) == THValueType.TABLE, "baseElement", baseElement) then
        if profileName == nil then
            return
        end
        if profileName == baseElement.profile then
            return baseElement
        end
        local elementList = baseElement.elements
        local numElements = elementList ~= nil and #elementList
        if numElements > 0 then
            for idx = 1, #elementList do
                local element = elementList[idx]
                if depth > 0 then
                    local foundElement = THUtils.getElementByProfileName(element, profileName, depth - 1)
                    if foundElement ~= nil then
                        return foundElement
                    end
                elseif profileName == element.profile then
                    return element
                end
            end
        end
    end
end