--[[
    RealGPSMapInit

    Initializes the RealGPS Map i3d

    @author: 		BayernGamers
    @date: 			11.03.2025
    @version:		2.0

    History:		v1.0 @15.07.2023 - initial implementation in FS 22
                    ------------------------------------------------------------------------------------------------------
                    v1.1 @16.07.2023 - update for Multi-PDA Support
                    ------------------------------------------------------------------------------------------------------
                    v1.2 @17.07.2023 - added glowShader to PDA
                    ------------------------------------------------------------------------------------------------------
                    v2.0 @11.03.2025 - convert and re-write for FS25
                    ------------------------------------------------------------------------------------------------------
    
    License:        Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
                        Attribution:
                            You must give appropriate credit to the original author when using this work.
                        No Derivatives:
                            You may not alter, transform, or build upon this work in any way.
                        Usage: 
                            The work may be used for personal and commercial purposes, provided it is not modified or adapted.
                        Additional Clause:
                            This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]
source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "RealGPSMapInit.lua")

RealGPSMapInit = {}
RealGPSMapInit.MOD_DIR = g_currentModDirectory
RealGPSMapInit.MOD_NAME = g_currentModName
RealGPSMapInit.MOD_SETTINGS_DIR = g_currentModSettingsDirectory
RealGPSMapInit.BASE_PATH = "realGPS.config"
RealGPSMapInit.XML_USER_SCHEMA = XMLSchema.new("realGPSUserConfig")
createFolder(RealGPSMapInit.MOD_SETTINGS_DIR)
createFolder(RealGPSMapInit.MOD_SETTINGS_DIR .. "pdaMap/")

g_xmlManager:addInitSchemaFunction(function()
    local schema = Mission00.xmlSchema
    schema:register(XMLValueType.STRING, "map#realGPSOverviewFilename", "RealGPS map overview filename")
end)

function getUtf8Codepoint(char)
    local b1, b2, b3, b4 = char:byte(1, 4)
    local codepoint = nil

    if b1 <= 0x7F then
        codepoint = b1
    elseif b1 >= 0xC2 and b1 <= 0xDF then
        codepoint = (b1 - 0xC0) * 0x40 + (b2 - 0x80)
    elseif b1 >= 0xE0 and b1 <= 0xEF then
        codepoint = (b1 - 0xE0) * 0x1000 + (b2 - 0x80) * 0x40 + (b3 - 0x80)
    elseif b1 >= 0xF0 and b1 <= 0xF4 then
        codepoint = (b1 - 0xF0) * 0x40000 + (b2 - 0x80) * 0x1000 + (b3 - 0x80) * 0x40 + (b4 - 0x80)
    end

    return codepoint
end

function convertUtf8ToUtf16(char)
    local codepoint = getUtf8Codepoint(char)

    if codepoint <= 0xFFFF then
        return string.char(codepoint % 256) .. string.char(math.floor(codepoint / 256))
    else
        codepoint = codepoint - 0x10000
        local highSurrogate = 0xD800 + math.floor(codepoint / 0x400)
        local lowSurrogate = 0xDC00 + (codepoint % 0x400)
        return string.char(lowSurrogate % 256) .. string.char(math.floor(lowSurrogate / 256))
                .. string.char(highSurrogate % 256) .. string.char(math.floor(highSurrogate / 256))
    end
end

function processCharacter(v, newDir)
    local utf16Char = convertUtf8ToUtf16(v)
    local char = ""
    if not string.match(v, "[.:%/0-9a-zA-Z_-]") then
        if utf16Char:byte(2) ~= nil then
            char = string.format("&#x%02X%02X", utf16Char:byte(2), utf16Char:byte(1)) .. ";"
        else
            char = string.format("&#x%02X%02X", 0, utf16Char:byte(1)) .. ";"
        end
    else
        char = v
    end
    
    newDir = newDir .. char
    return newDir
end

function convertUtf8ToCharArray(str)
    local charArray = {}
    local iStart = 0
    local strLen = str:len()

    local function bit(b)
        return 2 ^ (b - 1)
    end

    local function hasbit(w, b)
        return w % (b + b) >= b
    end

    local function checkMultiByte(i)
        if (iStart ~= 0) then
            charArray[#charArray + 1] = str:sub(iStart, i - 1)
            iStart = 0
        end
    end

    for i = 1, strLen do
        local b = str:byte(i)
        local multiStart = hasbit(b, bit(7)) and hasbit(b, bit(8))
        local multiTrail = not hasbit(b, bit(7)) and hasbit(b, bit(8))

        if (multiStart) then
            checkMultiByte(i)
            iStart = i
        elseif (not multiTrail) then
            checkMultiByte(i)
            charArray[#charArray + 1] = str:sub(i, i)
        end
    end

    checkMultiByte(strLen + 1)
    return charArray
end

function RealGPSMapInit.init()
    local schema = RealGPSMapInit.XML_USER_SCHEMA
    
    schema:setXMLSpecializationType("RealGPSUserConfig")

    schema:register(XMLValueType.INT, RealGPSMapInit.BASE_PATH .. "#version", "Version of the RealGPS Config", 2)
    schema:register(XMLValueType.BOOL, RealGPSMapInit.BASE_PATH .. "#pdaOverride", "Override the PDA map", false)
    schema:register(XMLValueType.STRING, RealGPSMapInit.BASE_PATH .. "#pdaFilename", "Filename of the PDA map", "")

    schema:setXMLSpecializationType()
end

function RealGPSMapInit.createI3dFile()
    local pdaMapFile = nil
    local mapTitle = g_currentMission.missionInfo.map.title

    mapTitle = mapTitle:gsub("%ä", "ae")
    mapTitle = mapTitle:gsub("%ö", "oe")
    mapTitle = mapTitle:gsub("%ü", "ue")

    local fileBasePath = RealGPSMapInit.MOD_SETTINGS_DIR .. mapTitle .. "/"
    local file = fileBasePath .. "RealGPS_config.xml"
    local configXML = XMLFile.loadIfExists("realGPSUserConfig", file, RealGPSMapInit.XML_USER_SCHEMA);

    createFolder(RealGPSMapInit.MOD_SETTINGS_DIR .. mapTitle)
    if configXML == nil then
        configXML = XMLFile.create("realGPSUserConfig", file, "realGPS", RealGPSMapInit.XML_USER_SCHEMA)
        configXML:setValue(RealGPSMapInit.BASE_PATH .. "#version", 2)
        configXML:setValue(RealGPSMapInit.BASE_PATH .. "#pdaOverride", false)
        configXML:setValue(RealGPSMapInit.BASE_PATH .. "#pdaFilename", "")
        configXML:save(true, false)
    else
        if configXML:getValue(RealGPSMapInit.BASE_PATH .. "#pdaOverride") and configXML:hasProperty(RealGPSMapInit.BASE_PATH .. "#version") and configXML:getValue(RealGPSMapInit.BASE_PATH .. "#version") >= 2 then
            pdaMapFile = fileBasePath .. configXML:getValue(RealGPSMapInit.BASE_PATH .. "#pdaFilename")

            if pdaMapFile:find(".png") then
                pdaMapFile = pdaMapFile:sub(0, pdaMapFile:len() - 4) .. ".dds"
            end
            pdaMapFile = ".." .. pdaMapFile:sub(RealGPSMapInit.MOD_SETTINGS_DIR:len(), pdaMapFile:len())
        elseif not configXML:hasProperty(RealGPSMapInit.BASE_PATH .. "#version") or configXML:getValue(RealGPSMapInit.BASE_PATH .. "#version") < 2 then
            log:printWarning("The RealGPS config file is outdated. Please only use RealGPS templates for FS25 (version 2.0 or higher)")
        end
    end

    if pdaMapFile == nil then
        for _, item in pairs(g_mapManager.maps) do
            if item.id == g_currentMission.missionInfo.map.id then
                local mapXMLFilename = item.mapXMLFilename
                
                if mapXMLFilename:find("$data") then
                    mapXMLFilename = getAppBasePath() .. mapXMLFilename:sub(2)
                else
                    mapXMLFilename = item.baseDirectory .. mapXMLFilename
                end
        
                local mapXML = XMLFile.loadIfExists("map", mapXMLFilename)

                local defaultOverview = mapXML:getString("map#imageFilename")
                local customOverview = mapXML:getString("map#realGPSOverviewFilename")

                if customOverview ~= nil then
                    pdaMapFile = customOverview
                else
                    pdaMapFile = defaultOverview
                end

                --RealGPSMapInit.worldSizeX = mapXML:getInt("map#width")
                --RealGPSMapInit.worldSizeZ = mapXML:getInt("map#height")
        
                if pdaMapFile:find("$data") then
                    pdaMapFile = getAppBasePath() .. pdaMapFile:sub(2)
                else
                    pdaMapFile = item.baseDirectory .. pdaMapFile
                end
        
                if pdaMapFile:find(".png") then
                    pdaMapFile = pdaMapFile:sub(0, pdaMapFile:len() - 4) .. ".dds"
                end
            end
        end
    end

    if pdaMapFile == nil then
        log:printError("Failed to find the PDA map")
        return
    end

    local charArr = convertUtf8ToCharArray(pdaMapFile)
    pdaMapFile = ""
    for _, v in pairs(charArr) do
        pdaMapFile = processCharacter(v, pdaMapFile)
    end

    local charArr = convertUtf8ToCharArray(RealGPSMapInit.MOD_DIR)
    local realGPSDir = ""
    for _, v in pairs(charArr) do
        realGPSDir = processCharacter(v, realGPSDir)
    end

    local i3dFileContent = [[
<?xml version="1.0" encoding="iso-8859-1"?>
<i3D name="pdaMap.i3d" version="1.6" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xsi:noNamespaceSchemaLocation="http://i3d.giants.ch/schema/i3d-1.6.xsd">
  <Asset>
    <Export program="GIANTS Editor 64bit" version="10.0.3"/>
  </Asset>

  <Files>
    <File fileId="4" filename="]] .. pdaMapFile .. [["/>
    <File fileId="3" filename="]] .. realGPSDir .. [[shaders/realGPSMapShader.xml"/>
    <File fileId="5" filename="]] .. realGPSDir .. [[textures/playerIcons.dds"/>
    <File fileId="6" filename="]] .. realGPSDir .. [[textures/playerIcons2.dds"/>
    <File fileId="7" filename="]] .. realGPSDir .. [[textures/playerIcons3.dds"/>
    <File fileId="8" filename="]] .. realGPSDir .. [[textures/playerIcons4.dds"/>
    <File fileId="9" filename="]] .. realGPSDir .. [[textures/gpsGuide.dds"/>
  </Files>


  <Materials>
    <Material name="realGPSMap_mat" materialId="5" diffuseColor="0.8 0.8 0.8 1" specularColor="0.501961 1 0.501961" customShaderId="3" shadingRate="2x4">
      <Custommap name="pdaMap" fileId="4"/>
      <Custommap name="playerIcons" fileId="5"/>
      <Custommap name="playerIcons2" fileId="6"/>
      <Custommap name="playerIcons3" fileId="7"/>
      <Custommap name="playerIcons4" fileId="8"/>
      <Custommap name="gpsGuides" fileId="9"/>
    </Material>
  </Materials>


  <Shapes externalShapesFile="pdaMap.i3d.shapes">
  </Shapes>

  <Scene>
    <Shape name="plane" shapeId="1" nodeId="8" castsShadows="true" receiveShadows="true" materialIds="5"/>
  </Scene>

</i3D>
]]

    local i3dShapesContent = {
        0x0A, 0x00, 0x60, 0x00, 0x97, 0x24, 0x36, 0xDF, 0xC0, 0x0A, 0xEE, 0xB2, 0xB8, 0x31, 0x8C, 0x18,
        0xEF, 0x28, 0x0C, 0xED, 0x47, 0x32, 0xF4, 0x89, 0x20, 0xBC, 0xB5, 0x66, 0x14, 0x5F, 0xE8, 0x8A,
        0xB7, 0x4E, 0x33, 0xB1, 0x4E, 0x15, 0xFC, 0xC0, 0xC6, 0x81, 0x58, 0x8A, 0xC0, 0xCB, 0xC6, 0x4E,
        0xBD, 0x13, 0xF0, 0x23, 0x39, 0xC2, 0x70, 0xE1, 0xD4, 0x19, 0x2A, 0x36, 0x66, 0xCC, 0xDE, 0xA8,
        0x04, 0x6A, 0x90, 0x2F, 0x39, 0x3A, 0xF9, 0xF2, 0x9B, 0xEF, 0x1C, 0xC0, 0xCB, 0x2E, 0xA4, 0xB0,
        0x11, 0xEB, 0x89, 0xEE, 0x3B, 0x67, 0x35, 0xCF, 0x64, 0x14, 0xE6, 0xC2, 0x56, 0xAF, 0x70, 0x2F,
        0x5E, 0x67, 0xFD, 0xC2, 0x25, 0x79, 0xD5, 0xFA, 0x58, 0x90, 0x47, 0x33, 0xD6, 0x72, 0x11, 0x8D,
        0x75, 0xAD, 0x01, 0x04, 0x3F, 0xB9, 0x2F, 0x79, 0xA1, 0x6E, 0xF3, 0x5B, 0xAA, 0xE9, 0x56, 0x59,
        0xEE, 0x00, 0x95, 0x21, 0x3D, 0x84, 0xCE, 0xDA, 0x5C, 0xBC, 0x2A, 0x15, 0x3B, 0x24, 0x2F, 0xDE,
        0xDE, 0x5D, 0x4D, 0x30, 0x00, 0x4E, 0xDE, 0xBA, 0x6B, 0xC4, 0x41, 0xA4, 0x15, 0xE3, 0x8F, 0xC4,
        0x07, 0xF6, 0xD9, 0x9D, 0x2C, 0xD2, 0x7B, 0xA0, 0x19, 0x7B, 0xA2, 0x59, 0x55, 0xAB, 0x42, 0xF2,
        0x4A, 0x1E, 0x6D, 0x0D, 0x57, 0x86, 0x25, 0xE9, 0x1D, 0xC0, 0x0C, 0x4E, 0x66, 0x90, 0x93, 0xEF,
        0xF9, 0x2B, 0x4E, 0x4B, 0xC9, 0x54, 0xC5, 0x3A, 0xD8, 0x05, 0x0A, 0x56, 0xBF, 0xC1, 0x55, 0xF9,
        0xAE, 0xBE, 0xFF, 0xB8, 0x35, 0x32, 0x5D, 0x4C, 0xE9, 0x49, 0x77, 0x9E, 0x4C, 0x7E, 0xB6, 0x7F,
        0x7D, 0x8B, 0xE6, 0x38, 0x27, 0x47, 0xEE, 0xA9, 0x6E, 0x8C, 0x44, 0x18, 0x65, 0xAC, 0x90, 0x33,
        0xC6, 0x30, 0xBE, 0xEE, 0x7D, 0x53, 0x4F, 0xD5, 0x23, 0x97, 0x3C, 0x1C, 0x72, 0x2C, 0xB8, 0x97,
        0xDD, 0x66, 0xBA, 0x76, 0xC0, 0x24, 0x53, 0x05, 0x55, 0x12, 0x46, 0xB7, 0x3B, 0xCD, 0xE5, 0xFD,
        0x4E, 0x7F, 0x09, 0xBA, 0xB9, 0x91, 0xA4, 0x8C, 0xB9, 0xD2, 0x6C, 0x14, 0xCF, 0x42, 0xA9, 0xBE,
        0x4F, 0x95, 0xE4, 0xDA, 0xCA, 0x45, 0x53, 0x85, 0xC5, 0x3A, 0x85, 0xDC, 0x61, 0x5A, 0xA8, 0xE4,
        0xB6, 0xE5, 0x97, 0x2C, 0x00, 0xA0, 0x0C, 0x1C
    }
    
    local i3dFile = io.open(RealGPSMapInit.MOD_SETTINGS_DIR .. "pdaMap/pdaMap.i3d", "w")
    if i3dFile ~= nil then
        i3dFile:write(i3dFileContent)
        i3dFile:close()
    end

    local i3dShapesFile = io.open(RealGPSMapInit.MOD_SETTINGS_DIR .. "pdaMap/pdaMap.i3d.shapes", "w")
    if i3dShapesFile ~= nil then
        local shapesFileContentString = ""
        for _, v in pairs(i3dShapesContent) do
            local charValue = string.char(v)
            shapesFileContentString = shapesFileContentString .. charValue
        end

        i3dShapesFile:write(shapesFileContentString)
        i3dShapesFile:close()
    end
end

FSBaseMission.loadMapFinished = Utils.appendedFunction(FSBaseMission.loadMapFinished, RealGPSMapInit.createI3dFile)
RealGPSMapInit.init()