--[[
    AutoTurnOffTurnLights

    Adds the ability to automatically turn off the turn lights when the steering wheel is turned back to the center position.

	@author: 		[LSFM] BayernGamers
	@date: 			12.11.2024
	@version:		1.0

	History:		v1.0 @02.01.2019 - initial implementation in FS25
                    ------------------------------------------------------------------------------------------------------

	
	License: 		This work is licensed under the Creative Commons Attribution-NoDerivs 4.0 International License (CC BY-ND 4.0).

					Terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage: 
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.

						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.

						Full License Text:
							The complete license text can be found at: https://creativecommons.org/licenses/by-nd/4.0/
]]

source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))

local log = LoggingUtil


AutoTurnOffTurnLights = {}
AutoTurnOffTurnLights.debugLevel = 1
AutoTurnOffTurnLights.debugLevelLow = 0

function AutoTurnOffTurnLights.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Drivable, specializations) and SpecializationUtil.hasSpecialization(Lights, specializations)
end

function AutoTurnOffTurnLights.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", AutoTurnOffTurnLights)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", AutoTurnOffTurnLights)
end

function AutoTurnOffTurnLights:onLoad(savegame)
    self.spec_autoTurnOffTurnLights = {}
    local spec_autoTurnOffTurnLights = self.spec_autoTurnOffTurnLights

    spec_autoTurnOffTurnLights.needsSteeringRight = false
    spec_autoTurnOffTurnLights.needsSteeringLeft = false
end

function AutoTurnOffTurnLights:onUpdate(dt)
    local spec_autoTurnOffTurnLights = self.spec_autoTurnOffTurnLights
    local spec_lights = self.spec_lights
    local spec_drivable = self.spec_drivable

    if self.getIsVehicleControlledByPlayer and self:getIsVehicleControlledByPlayer() and spec_drivable.steeringWheel and spec_drivable.steeringWheel.node ~= nil then
        local _, rotY, _ = getRotation(spec_drivable.steeringWheel.node)
        log.printDevInfo("Steering wheel rotation Y: " .. rotY, AutoTurnOffTurnLights.debugLevelLow, true, "AutoTurnOffTurnLights.lua")
    
        local isSteeringRight = rotY < -0.1
        local isSteeringLeft = rotY > 0.1
        log.printDevInfo("Steering right: " .. tostring(isSteeringRight), AutoTurnOffTurnLights.debugLevelLow, true, "AutoTurnOffTurnLights.lua")
        log.printDevInfo("Steering left: " .. tostring(isSteeringLeft), AutoTurnOffTurnLights.debugLevelLow, true, "AutoTurnOffTurnLights.lua")
    
        local turnLightRightActive = spec_lights.turnLightState == Lights.TURNLIGHT_RIGHT
        local turnLightLeftActive = spec_lights.turnLightState == Lights.TURNLIGHT_LEFT
    
        if spec_drivable.needsSteeringLeft or spec_drivable.needsSteeringRight then
            local allowAutoTurnOffTurnLights = (spec_drivable.needsSteeringLeft and rotY <= 0.001) or (spec_drivable.needsSteeringRight and rotY >= -0.001)
            
            if allowAutoTurnOffTurnLights then
                self:setTurnLightState(Lights.TURNLIGHT_OFF)
                spec_drivable.needsSteeringLeft = false
                spec_drivable.needsSteeringRight = false
            end
        else
            spec_drivable.needsSteeringLeft = turnLightLeftActive and isSteeringLeft
            spec_drivable.needsSteeringRight = turnLightRightActive and isSteeringRight
        end
    end
end