--[[
    LockSteeringAxle

    Enables the opportunity to lock the steering axle of a vehicle.

	@author: 		[LSFM] BayernGamers
	@date: 			13.11.2024
	@version:		1.0

	History:		v1.0 @13.11.2024 - initial implementation in FS25
                    ------------------------------------------------------------------------------------------------------

	
	License: 		This work is licensed under the Creative Commons Attribution-NoDerivs 4.0 International License (CC BY-ND 4.0).

					Terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage: 
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.

						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.

						Full License Text:
							The complete license text can be found at: https://creativecommons.org/licenses/by-nd/4.0/
]]

source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))
source(Utils.getFilename("scripts/utils/AdditionalMathUtil.lua", g_currentModDirectory))
source(Utils.getFilename("scripts/events/LockSteeringAxleEvent.lua", g_currentModDirectory))
source(Utils.getFilename("scripts/events/LockSteeringAxleAutoEvent.lua", g_currentModDirectory))

local log = LoggingUtil

LockSteeringAxle = {}
LockSteeringAxle.debugLevel = 1
LockSteeringAxle.debugLevelLow = 0
LockSteeringAxle.MOD_DIRECTORY = g_currentModDirectory
LockSteeringAxle.MOD_NAME = g_currentModName
LockSteeringAxle.steeringAxleAngleScaleKey = "vehicle.attachable.steeringAxleAngleScale"

function LockSteeringAxle.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Attachable, specializations)
end

function LockSteeringAxle.initSpecialization()
	local schemaSavegame = Vehicle.xmlSchemaSavegame

	schemaSavegame:register(XMLValueType.BOOL, "vehicles.vehicle(?)." .. LockSteeringAxle.MOD_NAME .. ".lockSteeringAxle#lockSteeringAxle", "Steering Axle is locked.", false)
	schemaSavegame:register(XMLValueType.BOOL, "vehicles.vehicle(?)." .. LockSteeringAxle.MOD_NAME .. ".lockSteeringAxle#lockSteeringAxleAuto", "Steering Axle is automatically locked.", false)
	log.printDevInfo("Registered 'lockSteeringAxle' xmlPaths", 1, true, "LockSteeringAxle.lua") 
end

function LockSteeringAxle.registerEventListeners(vehicleType)
	SpecializationUtil.registerEventListener(vehicleType, "onLoad", LockSteeringAxle)
	SpecializationUtil.registerEventListener(vehicleType, "onUpdate", LockSteeringAxle)
	SpecializationUtil.registerEventListener(vehicleType, "saveToXMLFile", LockSteeringAxle)
	SpecializationUtil.registerEventListener(vehicleType, "onWriteStream", LockSteeringAxle)
	SpecializationUtil.registerEventListener(vehicleType, "onReadStream", LockSteeringAxle)
	SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", LockSteeringAxle)
end

function LockSteeringAxle.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "toggleLockSteeringAxle", LockSteeringAxle.toggleLockSteeringAxle)
    SpecializationUtil.registerFunction(vehicleType, "toggleLockSteeringAxleAuto", LockSteeringAxle.toggleLockSteeringAxleAuto)
    SpecializationUtil.registerFunction(vehicleType, "updateToggleLockSteeringAxleActionEvent", LockSteeringAxle.updateToggleLockSteeringAxleActionEvent)
	SpecializationUtil.registerFunction(vehicleType, "updateToggleLockSteeringAxleAutoActionEvent", LockSteeringAxle.updateToggleLockSteeringAxleAutoActionEvent)
end

function LockSteeringAxle:onLoad(savegame)
    log.printDevInfo("Load Self: " .. tostring(self:getName()), LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")

    self.spec_lockSteeringAxle = {}
	local spec_lockSteeringAxle = self.spec_lockSteeringAxle
	local spec_wheels = self.spec_wheels
	local spec_attachable = self.spec_attachable

    spec_lockSteeringAxle.lockSteeringAxle = false
	spec_lockSteeringAxle.lockSteeringAxleAuto = false
    spec_lockSteeringAxle.hasSteeringAxle = self.xmlFile:hasProperty(LockSteeringAxle.steeringAxleAngleScaleKey)


    if spec_lockSteeringAxle.hasSteeringAxle and savegame ~= nil then
        log.printDevInfo("Loading 'lockSteeringAxle' xmlPaths '" .. savegame.key .. "#lockSteeringAxle'", LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")
        spec_lockSteeringAxle.lockSteeringAxle = savegame.xmlFile:getValue(savegame.key .. "." .. LockSteeringAxle.MOD_NAME .. ".lockSteeringAxle#lockSteeringAxle", spec_lockSteeringAxle.lockSteeringAxle)
        log.printDevInfo("lockSteeringAxle: " .. tostring(spec_lockSteeringAxle.lockSteeringAxle), LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")

		spec_lockSteeringAxle.lockSteeringAxleAuto = savegame.xmlFile:getValue(savegame.key .. "." .. LockSteeringAxle.MOD_NAME .. ".lockSteeringAxle#lockSteeringAxleAuto", spec_lockSteeringAxle.lockSteeringAxleAuto)
		log.printDevInfo("lockSteeringAxleAuto: " .. tostring(spec_lockSteeringAxle.lockSteeringAxleAuto), LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")
	end
end

function LockSteeringAxle:onRegisterActionEvents(isActiveForInput)
    local spec_lockSteeringAxle = self.spec_lockSteeringAxle
    spec_lockSteeringAxle.actionEvents = {}

    if isActiveForInput then
        local _, actionEventId = self:addActionEvent(spec_lockSteeringAxle.actionEvents, InputAction.TOGGLE_STEERING_AXLE_LOCK, self, LockSteeringAxle.toggleLockSteeringAxle, false, true, false, true)
        
        g_inputBinding:setActionEventTextVisibility(actionEventId, true)
        g_inputBinding:setActionEventText(actionEventId, g_i18n:getText("action_LOCK_STEERING_AXLE"))
        g_inputBinding:setActionEventTextPriority(actionEventId, GS_PRIO_LOW)
		log.printDevInfo("ActionEventId: " .. tostring(actionEventId), LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")

		local valid, actionEventId2 = self:addActionEvent(spec_lockSteeringAxle.actionEvents, InputAction.TOGGLE_AUTO_STEERING_AXLE_LOCK, self, LockSteeringAxle.toggleLockSteeringAxleAuto, false, true, false, true)

		g_inputBinding:setActionEventTextVisibility(actionEventId2, true)
        g_inputBinding:setActionEventText(actionEventId2, g_i18n:getText("action_LOCK_STEERING_AXLE_AUTO"))
        g_inputBinding:setActionEventTextPriority(actionEventId2, GS_PRIO_LOW)
		log.printDevInfo("Valid: " .. tostring(valid), LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")
		log.printDevInfo("ActionEventId2: " .. tostring(actionEventId2), LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")

		table.insert(spec_lockSteeringAxle.actionEvents, actionEventId)
		table.insert(spec_lockSteeringAxle.actionEvents, actionEventId2)
    end
end

function LockSteeringAxle:onWriteStream(streamId, connection)
	if not connection:getIsServer() then
		local spec_lockSteeringAxle = self.spec_lockSteeringAxle
		
		if spec_lockSteeringAxle.hasSteeringAxle and spec_lockSteeringAxle.lockSteeringAxle ~= nil then
			streamWriteBool(streamId, spec_lockSteeringAxle.lockSteeringAxle)
		end
	end
end

function LockSteeringAxle:onReadStream(streamId, connection)
	if connection:getIsServer() then
		local spec_lockSteeringAxle = self.spec_lockSteeringAxle
		
		if spec_lockSteeringAxle.hasSteeringAxle and spec_lockSteeringAxle.lockSteeringAxle ~= nil then	
			spec_lockSteeringAxle.lockSteeringAxle = streamReadBool(streamId)
		end
	end
end

function LockSteeringAxle:saveToXMLFile(xmlFile, key, usedModNames)
	local spec_lockSteeringAxle = self.spec_lockSteeringAxle
	
	if spec_lockSteeringAxle.hasSteeringAxle then
		log.printDevInfo("Saving 'lockSteeringAxle' to xmlPath '" .. key .. "#lockSteeringAxle'", LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")
		log.printDevInfo("lockSteeringAxle: " .. tostring(spec_lockSteeringAxle.lockSteeringAxle), LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")
		xmlFile:setValue(key .. "#lockSteeringAxle", spec_lockSteeringAxle.lockSteeringAxle)
		xmlFile:setValue(key .. "#lockSteeringAxleAuto", spec_lockSteeringAxle.lockSteeringAxleAuto)
	end
end

function LockSteeringAxle:onUpdate(dt, isActiveForInput, isActiveForInputIgnoreSelection, isSelected)
	local spec = self.spec_attachable
    local spec_lockSteeringAxle = self.spec_lockSteeringAxle
	local yRot = nil

    log.printDevInfo("Update Self: " .. tostring(self:getName()), LockSteeringAxle.debugLevelLow, true, "LockSteeringAxle.lua")
    log.printDevInfo("spec_LockSteeringAxle: " .. tostring(spec_lockSteeringAxle), LockSteeringAxle.debugLevelLow, true, "LockSteeringAxle.lua")

    spec.updateSteeringAxleAngle = not spec_lockSteeringAxle.lockSteeringAxle
	spec.steeringAxleUpdateBackwards = not spec_lockSteeringAxle.lockSteeringAxleAuto

	self:updateToggleLockSteeringAxleActionEvent()
	self:updateToggleLockSteeringAxleAutoActionEvent()

    if spec_lockSteeringAxle.lockSteeringAxle == true then
        local currentAngle = spec.steeringAxleAngle
        local targetAngle = 0
        local speed = 0.001

        if currentAngle > targetAngle then
            spec.steeringAxleAngle = math.max(currentAngle - speed * dt, targetAngle)
        elseif currentAngle < targetAngle then
            spec.steeringAxleAngle = math.min(currentAngle + speed * dt, targetAngle)
        end

        if spec.steeringAxleTargetNode ~= nil then
            setRotation(spec.steeringAxleTargetNode, 0, spec.steeringAxleAngle, 0)
            self:setMovingToolDirty(spec.steeringAxleTargetNode)
        end
        return
    end
end

function LockSteeringAxle:updateToggleLockSteeringAxleActionEvent()
    local spec_lockSteeringAxle = self.spec_lockSteeringAxle

    if spec_lockSteeringAxle.actionEvents ~= nil then
        local actionEventId = spec_lockSteeringAxle.actionEvents[1]

        if actionEventId ~= nil then
            local text = g_i18n:getText("action_LOCK_STEERING_AXLE")

            if spec_lockSteeringAxle.lockSteeringAxle then
                text = g_i18n:getText("action_UNLOCK_STEERING_AXLE")
            end

            g_inputBinding:setActionEventText(actionEventId, text)
        end
    end
end

function LockSteeringAxle:updateToggleLockSteeringAxleAutoActionEvent()
	local spec_lockSteeringAxle = self.spec_lockSteeringAxle

    if spec_lockSteeringAxle.actionEvents ~= nil then
        local actionEventId = spec_lockSteeringAxle.actionEvents[2]

        if actionEventId ~= nil then
            local text = g_i18n:getText("action_LOCK_STEERING_AXLE_AUTO")

            if spec_lockSteeringAxle.lockSteeringAxleAuto then
                text = g_i18n:getText("action_UNLOCK_STEERING_AXLE_AUTO")
            end

            g_inputBinding:setActionEventText(actionEventId, text)
        end
    end	
end

function LockSteeringAxle:toggleLockSteeringAxle()
	log.printDevInfo("Toggle Lock Steering Axle", LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")
    local spec_lockSteeringAxle = self.spec_lockSteeringAxle

    if spec_lockSteeringAxle.lockSteeringAxle ~= nil then
		log.printDevInfo("LockSteeringAxle: " .. tostring(spec_lockSteeringAxle.lockSteeringAxle), LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")
        spec_lockSteeringAxle.lockSteeringAxle = not spec_lockSteeringAxle.lockSteeringAxle

		log.printDevInfo("Updated LockSteeringAxle: " .. tostring(spec_lockSteeringAxle.lockSteeringAxle), LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")

        if g_server ~= nil then
			log.printDevInfo("Broadcasting LockSteeringAxleEvent", LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")
            g_server:broadcastEvent(LockSteeringAxleEvent.new(self, spec_lockSteeringAxle.lockSteeringAxle), nil, nil, self)
        else
			log.printDevInfo("Sending LockSteeringAxleEvent", LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")
            g_client:getServerConnection():sendEvent(LockSteeringAxleEvent.new(self, spec_lockSteeringAxle.lockSteeringAxle))
        end
    end
end

function LockSteeringAxle:toggleLockSteeringAxleAuto()
	log.printDevInfo("Toggle Lock Steering Axle Auto", LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")
	local spec_lockSteeringAxle = self.spec_lockSteeringAxle

	if spec_lockSteeringAxle.lockSteeringAxleAuto ~= nil then
		log.printDevInfo("LockSteeringAxleAuto: " .. tostring(spec_lockSteeringAxle.lockSteeringAxleAuto), LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")
		spec_lockSteeringAxle.lockSteeringAxleAuto = not spec_lockSteeringAxle.lockSteeringAxleAuto
		self:updateToggleLockSteeringAxleAutoActionEvent()

		log.printDevInfo("Updated LockSteeringAxleAuto: " .. tostring(spec_lockSteeringAxle.lockSteeringAxleAuto), LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")

		if g_server ~= nil then
			log.printDevInfo("Broadcasting LockSteeringAxleAutoEvent", LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")
			g_server:broadcastEvent(LockSteeringAxleAutoEvent.new(self, spec_lockSteeringAxle.lockSteeringAxleAuto), nil, nil, self)
		else
			log.printDevInfo("Sending LockSteeringAxleAutoEvent", LockSteeringAxle.debugLevel, true, "LockSteeringAxle.lua")
			g_client:getServerConnection():sendEvent(LockSteeringAxleAutoEvent.new(self, spec_lockSteeringAxle.lockSteeringAxleAuto))
		end
	end
end