--[[
    RegisterSpecialization

    Registers a specialization for a vehicle type. The script checks the prerequisites for the specialization and adds it to the vehicle type if it is not already present.

	@author: 		[LSFM] BayernGamers
	@date: 			12.11.2024
	@version:		1.0

	History:		v1.0 @12.11.2024 - initial implementation in FS25
                    ------------------------------------------------------------------------------------------------------

	
	License: 		This work is licensed under the Creative Commons Attribution-NoDerivs 4.0 International License (CC BY-ND 4.0).

					Terms:
						Attribution:
							You must give appropriate credit to the original author when using this work.
						No Derivatives:
							You may not alter, transform, or build upon this work in any way.
						Usage: 
							The work may be used for personal and commercial purposes, provided it is not modified or adapted.

						Additional Clause:
							This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.

						Full License Text:
							The complete license text can be found at: https://creativecommons.org/licenses/by-nd/4.0/
]]

source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))

local log = LoggingUtil

RegisterSpecialization = {}
RegisterSpecialization.debugLevel = 1
RegisterSpecialization.debugLevelLow = 0
RegisterSpecialization.MOD_DIRECTORY = g_currentModDirectory
RegisterSpecialization.MOD_NAME = g_currentModName
RegisterSpecialization.XML_KEY = "modDesc.registerSpecializations.specialization"
RegisterSpecialization.modDescPath = RegisterSpecialization.MOD_DIRECTORY .. "modDesc.xml"
RegisterSpecialization.specializations = {}

function RegisterSpecialization.getClassObject(className)
    local parts = string.split(className, ".")
    local currentTable = _G[parts[1]]

    if type(currentTable) ~= "table" then
        return nil
    end

    for i = 2, #parts do
        currentTable = currentTable[parts[i]]

        if type(currentTable) ~= "table" then
            return nil
        end
    end

    return currentTable
end

local modDesc = XMLFile.load("ModFile", RegisterSpecialization.modDescPath)

modDesc:iterate(RegisterSpecialization.XML_KEY, function (_, key)
    log.printDevInfo("Loading Specialization data for key '" .. key .. "'", RegisterSpecialization.debugLevel, true, "RegisterSpecialization.lua")
    local specName = modDesc:getString(key .. "#name")
    local typeName = specName
    local className = modDesc:getString(key .. "#className")
    local fileName = modDesc:getString(key .. "#fileName")

    if specName == nil or specName == "" then
        log.printXMLError(modDesc, "Missing value for 'name' in specialization" .. key)
    end

    if className == nil or className == "" then
        log.printXMLError(modDesc, "Missing value for 'className' in specialization" .. key)
    end

    if fileName == nil or fileName == "" then
        log.printXMLError(modDesc, "Missing value for 'fileName' in specialization" .. key)
    end

    local filePath = Utils.getFilename(fileName, RegisterSpecialization.MOD_DIRECTORY)

    if filePath == nil or filePath == "" then
        log.printXMLError(modDesc, "Invalid value for 'fileName' in specialization" .. key)
    end

    source(filePath)
    local class = RegisterSpecialization.getClassObject(className)

    if class == nil then
        log.printError("Class '" .. className .. "' not found in file '" .. fileName .. "'", true, "RegisterSpecialization.lua")
    end

    if class.prerequisitesPresent == nil then
        log.printError("Function 'prerequisitesPresent' not found in class '" .. className .. "'", true, "RegisterSpecialization.lua")
    end
    
    if g_specializationManager:getSpecializationByName(specName) == nil then
        g_specializationManager:addSpecialization(typeName, className, filePath, nil)
        log.printInfo("Added specialization '" .. typeName .. "'", true, "RegisterSpecialization.lua")
    end

    for vehicleTypeName, vehicleType in pairs(g_vehicleTypeManager.types) do
        if vehicleType ~= nil and class.prerequisitesPresent(vehicleType.specializations) then
            g_vehicleTypeManager:addSpecialization(vehicleTypeName, typeName)
            log.printInfo("Added specialization '" .. typeName .. "' to " .. "'" .. vehicleTypeName .. "'", true, "RegisterSpecialization.lua")
        end
    end
end)